package com.codingame.game;
import java.util.HashMap;
import java.util.List;
import java.util.Random;

import com.codingame.game.ataxx.*;
import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.GameManager;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.endscreen.EndScreenModule;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.toggle.ToggleModule;
import com.google.inject.Inject;

import javax.swing.plaf.synth.SynthEditorPaneUI;

public class Referee extends AbstractReferee {
    @Inject private MultiplayerGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphics;
    @Inject private ToggleModule toggleModule;
    @Inject private EndScreenModule endScreenModule;

    Random rand;
    Board board;
    Viewer viewer;
    Player currentPlayer;
    final int max_turns = 300;

    int pieces[] = new int[2];

    List<Action> actions;
    HashMap<String, Integer> repetitions;

    @Override
    public void init() {
        //-9142635365175899773
        //System.err.println(gameManager.getSeed());
        repetitions = new HashMap<>();
        gameManager.setFirstTurnMaxTime(1000);
        gameManager.setTurnMaxTime(100);
        gameManager.setMaxTurns(600);
        rand = new Random(gameManager.getSeed());
        board = new Board(7, gameManager.getSeed());
        viewer = new Viewer(graphics, board, gameManager, toggleModule);

        currentPlayer = gameManager.getPlayer(0);
        gameManager.getPlayer(0).opponent = gameManager.getPlayer(1);
        gameManager.getPlayer(1).opponent = gameManager.getPlayer(0);

        actions = board.getActions(currentPlayer.getIndex());
        currentPlayer.lastMoves = "";

        String hc = board.hash();
        repetitions.put(hc, 1);
        board.current_player ^= 1;
        hc = board.hash();
        repetitions.put(hc, 1);
        board.current_player ^= 1;
    }

    @Override
    public void gameTurn(int turn) {
        sendInputs(turn);
        currentPlayer.execute();

        try {
            String[] outputs = currentPlayer.getOutputs().get(0).split("\\s");
            outputs[0] = outputs[0].toLowerCase();

            if (outputs.length > 1) {
                outputs[1] = outputs[1].substring(0, Math.min(outputs[1].length(), 16));
                viewer.playerUIS[currentPlayer.getIndex()].msg.setText(outputs[1]);
            } else {
                viewer.playerUIS[currentPlayer.getIndex()].msg.setText("");
            }

            boolean found = false;
            if (outputs[0].equals("random")) {
                found = true;

                int r = rand.nextInt(actions.size());
                currentPlayer.lastAction = actions.get(r).toString();
                currentPlayer.lastMoves += actions.get(r).toString();
                board.applyAction(actions.get(r), currentPlayer.getIndex(), viewer);
            }
            else {
                for (Action action : actions) {
                    if (action.toString().equals(outputs[0])) {
                        found = true;
                        currentPlayer.lastAction = action.toString();
                        currentPlayer.lastMoves += action.toString();

                        board.applyAction(action, currentPlayer.getIndex(), viewer);
                        break;
                    }
                }
            }

            if(!found) {
                throw new InvalidAction("Action was not valid!");
            }

            viewer.playerUIS[currentPlayer.getIndex()].action.setText(currentPlayer.lastAction);
            graphics.commitEntityState(0, viewer.playerUIS[currentPlayer.getIndex()].action, viewer.playerUIS[currentPlayer.getIndex()].msg);

            pieces[0] = pieces[1] = 0;
            for (Unit u : board.units) {
                pieces[u.owner]++;
            }
            for (int i = 0; i < 2; ++i) {
                viewer.playerUIS[i].score.setText(Integer.toString(pieces[i]));
            }

            String hc = board.hash();
            if (repetitions.containsKey(hc)) {
                repetitions.replace(hc, repetitions.get(hc) + 1);
                if (repetitions.get(hc) >= 3) {
                    gameManager.addToGameSummary(GameManager.formatErrorMessage("Draw by repetition!"));
                    gameManager.endGame();
                    return;
                }
            }
            else {
                repetitions.put(hc, 1);
            }
           // System.err.println(board.hash());

            if (board.isGameOver()) {
                pieces[0] = pieces[1] = 0;
                for (Unit u : board.units) {
                    pieces[u.owner]++;
                }
                if (pieces[0] > pieces[1]) {
                    gameManager.getPlayer(0).setScore(1);
                }
                else if (pieces[0] < pieces[1]) {
                    gameManager.getPlayer(1).setScore(1);
                }
                gameManager.endGame();
                return;
            }
            else if (turn == max_turns) { // If it reaches max turns it's a draw
                pieces[0] = pieces[1] = 0;
                gameManager.addToGameSummary(GameManager.formatErrorMessage("Draw, turn limit reached!"));
                gameManager.endGame();
                return;
            }
            else {
                List<Action> next_actions = board.getActions(currentPlayer.opponent.getIndex());
                // If the next player has a move
                if (next_actions.size() > 0) {
                    actions = next_actions;
                    currentPlayer = currentPlayer.opponent;
                    board.current_player ^= 1;
                    currentPlayer.lastMoves = "";
                }
                else {
                    actions = board.getActions(currentPlayer.getIndex());
                    currentPlayer.lastMoves += ";";
                    currentPlayer.opponent.lastAction = "pass";
                    currentPlayer.opponent.lastMoves = "pass";

                    viewer.playerUIS[currentPlayer.opponent.getIndex()].action.setText(currentPlayer.opponent.lastAction);
                }
            }

        } catch (TimeoutException e) {
            gameManager.addToGameSummary(GameManager.formatErrorMessage(currentPlayer.getNicknameToken() + " did not output in time!"));
            currentPlayer.deactivate(currentPlayer.getNicknameToken() + " timeout.");
            currentPlayer.setScore(-1);
            gameManager.endGame();
            return;
        } catch (NumberFormatException | ArrayIndexOutOfBoundsException | InvalidAction e) {
            gameManager.addToGameSummary(GameManager.formatErrorMessage(currentPlayer.getNicknameToken() + " made an invalid action!"));
            currentPlayer.deactivate(currentPlayer.getNicknameToken() + " made an invalid action.");
            currentPlayer.setScore(-1);
            gameManager.endGame();
            return;
        }
    }

    public void sendInputs(int turn) {
        if (turn <= 2) {
            // ID
            currentPlayer.sendInputLine(Integer.toString(currentPlayer.getIndex()));
            currentPlayer.sendInputLine(board.board_config);
        }

        for(int y = 0; y < 7; ++y) {
            String s = "";
            for (int x = 0; x < 7; ++x) {
                Cell cell = board.cells[7 - y - 1][x];
                if (cell.wall) {
                    s += "#";
                } else if (cell.unit != null) {
                    s += cell.unit.owner == 0 ? '0' : '1';
                } else {
                    s += ".";
                }
            }
            currentPlayer.sendInputLine(s);
        }

        currentPlayer.sendInputLine(currentPlayer.opponent.lastMoves);
        currentPlayer.sendInputLine(Integer.toString(actions.size()));
    }

    @Override
    public void onEnd() {
        int[] scores = { gameManager.getPlayer(0).getScore(), gameManager.getPlayer(1).getScore() };
        String[] text = new String[2];
        if(scores[0] > scores[1]) {
            text[0] = "Won " + pieces[0];
            text[1] = "Lost " + pieces[1];
            gameManager.addTooltip(gameManager.getPlayer(0), gameManager.getPlayer(0).getNicknameToken() + " won the match");
        } else if(scores[1] > scores[0]) {
            text[0] = "Lost " + pieces[0];
            text[1] = "Won " + pieces[1];
            gameManager.addTooltip(gameManager.getPlayer(1), gameManager.getPlayer(1).getNicknameToken() + " won the match");
        } else {
            text[0] = "Draw";
            text[1] = "Draw";
        }
        endScreenModule.setScores(scores, text);
    }
}
