package com.codingame.game;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.SoloGameManager;
import com.codingame.gameengine.module.entities.Curve;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.entities.Sprite;
import com.google.inject.Inject;

public class Referee extends AbstractReferee {
    @Inject private SoloGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;

		private Coord paddlePosition = new Coord(4, 0);
		private Coord itemPosition;
		private boolean isItemRed;
    
		private Sprite paddleSprite;
		private Sprite redSprite;
		private Sprite greenSprite;
    
		private Integer[] itemPositions;
		private int itemPositionsIndex = 0;

    @Override
    public void init() {
        gameManager.setFrameDuration(500);
        
        // Draw background
        graphicEntityModule.createSprite().setImage(Constants.BACKGROUND_SPRITE);

        itemPositions = Arrays.stream(gameManager.getTestCaseInput().get(0).split(" "))
            .map(s -> Integer.valueOf(s))
            .toArray(size -> new Integer[size]);
				itemPosition = new Coord(itemPositions[itemPositionsIndex].intValue() % Constants.MAX_X, Constants.MAX_Y - 1);
				isItemRed = itemPositions[itemPositionsIndex].intValue() >= Constants.MAX_X;
        
        paddleSprite = graphicEntityModule.createSprite().setImage(Constants.PADDLE_SPRITE)
            .setX(paddlePosition.x * Constants.CELL_SIZE_X + Constants.CELL_OFFSET_X)
            .setY((Constants.MAX_Y - paddlePosition.y) * Constants.CELL_SIZE_Y + Constants.CELL_OFFSET_Y)
            .setAnchor(.5)
            .setZIndex(1);

				redSprite = graphicEntityModule.createSprite().setImage(Constants.RED_SPRITE);
				greenSprite = graphicEntityModule.createSprite().setImage(Constants.GREEN_SPRITE);

        updateSpritePosition();					
    }

		private void updateSpritePosition() {
			redSprite.setX(itemPosition.x * Constants.CELL_SIZE_X + Constants.CELL_OFFSET_X, Curve.IMMEDIATE)
			    .setY(getItemY(), Curve.IMMEDIATE)
			    .setAnchor(.5)
			    .setZIndex(2);			
			greenSprite.setX(itemPosition.x * Constants.CELL_SIZE_X + Constants.CELL_OFFSET_X, Curve.IMMEDIATE)
			    .setY(getItemY(), Curve.IMMEDIATE)
			    .setAnchor(.5)
			    .setZIndex(2);
			if (isItemRed ) {
				greenSprite.setY(-2000, Curve.IMMEDIATE);
			} else {
				redSprite.setY(-2000, Curve.IMMEDIATE);
			}  
		}

		private int getItemY() {
			return (Constants.MAX_Y - itemPosition.y) * Constants.CELL_SIZE_Y + Constants.CELL_OFFSET_Y;
		}

    @Override
    public void gameTurn(int turn) {
			gameManager.getPlayer().sendInputLine(paddlePosition.toString());
      gameManager.getPlayer().sendInputLine(itemPosition.toString() + " " + (isItemRed ? "1" : "0"));

      gameManager.getPlayer().execute();

        try {
            List<String> outputs = gameManager.getPlayer().getOutputs();

            String output = checkOutput(outputs);

            if (output != null) {
                Action action = Action.valueOf(output.toUpperCase());
								Coord actionCoord = Constants.ACTION_MAP.get(action);                
                Coord candidateCoord = paddlePosition.add(actionCoord);
								if(candidateCoord.x >= 0 && candidateCoord.x < Constants.MAX_X){
									paddlePosition = candidateCoord;
								}
            }
        } catch (TimeoutException e) {
            gameManager.loseGame("Timeout!");
        }
				itemPosition = itemPosition.add(Coord.DOWN);
				if (itemPosition.y <= 0) {
					if (isItemRed == (paddlePosition.x == itemPosition.x)){
						gameManager.loseGame("Lost!");
					} else {
						itemPositionsIndex++;
						if (itemPositionsIndex >= itemPositions.length){
							gameManager.winGame("Congrats!");
						} else {
							itemPosition = new Coord(itemPositions[itemPositionsIndex].intValue() % Constants.MAX_X, Constants.MAX_Y - 1);
							isItemRed = itemPositions[itemPositionsIndex].intValue() >= Constants.MAX_X;
							updateSpritePosition();
						}
					}
				}        
        updateView();
    }

    private void updateView() {
			Sprite item = isItemRed? redSprite : greenSprite;
			if (item.getY() !=  getItemY()) {				
				item.setY(getItemY(), Curve.LINEAR);
			}    
			paddleSprite.setX(paddlePosition.x * Constants.CELL_SIZE_X + Constants.CELL_OFFSET_X, Curve.IMMEDIATE);
    }

    private String checkOutput(List<String> outputs) {
        if (outputs.size() != 1) {
            gameManager.loseGame("You did not send 1 output in your turn.");
        } else {
            String output = outputs.get(0);
            if (!Arrays.asList(Constants.ACTIONS).contains(output)) {
                gameManager
                    .loseGame(
                        String.format(
                            "Expected output: %s but received %s",
                            Arrays.asList(Constants.ACTIONS).stream().collect(Collectors.joining(" | ")),
                            output
                        )
                    );
            } else {
                return output;
            }
        }
        return null;
    }    
}
