package com.codingame.game;

import com.codingame.game.exceptions.BombException;
import com.codingame.game.exceptions.OutOfBoundException;
import com.codingame.game.exceptions.SameCoordinatesException;
import com.codingame.game.exceptions.SameIntegerException;
import com.codingame.game.heap.BinarySearchTree;
import com.codingame.gameengine.core.AbstractPlayer;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.SoloGameManager;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gui.GuiManager;
import com.google.inject.Inject;

import java.util.HashSet;
import java.util.List;

public class Referee extends AbstractReferee {
    @Inject private SoloGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;

    /**
     * Manager of the graphical interface
     */
    private GuiManager guiManager;

    /**
     * A binary tree, built by the player to solve the puzzle.
     */
    private BinarySearchTree binarySearchTree;

    /**
     * The maximum integer the user can put in the tree. It should add all or part of the integers between 1 and n.
     */
    private int n;

    @Override
    public void init() {
        gameManager.setFrameDuration(400);

        // Index of the current readed input line
        int inputIndex = 0;

        // Init the parameters
        List<String> testCase = gameManager.getTestCaseInput();

        // First line, containing the width, the height, n, and the numbers of bombs and of goals
        String[] line = testCase.get(inputIndex++).split(" ");

        int width = Integer.valueOf(line[0]);
        int height = Integer.valueOf(line[1]);
        binarySearchTree = new BinarySearchTree(width);
        n = Integer.valueOf(line[2]);

        // Set the bombs
        int bombsSize = Integer.valueOf(line[3]);
        int[][] bombs = new int[bombsSize][2];
        for(int i = 0; i < bombsSize; i++) {
            String[] lineAr = testCase.get(inputIndex++).split(" ");
            int bombX = Integer.valueOf(lineAr[0]);
            int bombY = Integer.valueOf(lineAr[1]);
            bombs[i][0] = bombX;
            bombs[i][1] = bombY;
        }
        binarySearchTree.setBombs(bombs);

        // Set the goals
        int goalsSize = Integer.valueOf(line[4]);
        int maxGoalY = 0; // Maximum y-coordinate of the goals, for the GUI
        int[][] goals = new int[goalsSize][2];
        for(int i = 0; i < goalsSize; i++) {
            String[] lineAr = testCase.get(inputIndex++).split(" ");
            int goalX = Integer.valueOf(lineAr[0]);
            int goalY = Integer.valueOf(lineAr[1]);
            maxGoalY = Math.max(maxGoalY, goalY);
            goals[i][0] = goalX;
            goals[i][1] = goalY;
        }
        binarySearchTree.setGoals(goals);

        // Set the GUI
        guiManager = new GuiManager(graphicEntityModule, width, maxGoalY + 1);
        guiManager.initGui();

        for(int i = 0; i < bombsSize; i++)
            guiManager.addBomb(bombs[i][0], bombs[i][1]);

        for(int i = 0; i < goalsSize; i++)
            guiManager.addGoal(goals[i][0], goals[i][1]);

        // Send all the inputs to the user
        for (String s : testCase) gameManager.getPlayer().sendInputLine(s);
    }

    @Override
    public void gameTurn(int turn) {
        try {
            // Let the user play one turn
            gameManager.getPlayer().execute();

            // Get the output
            List<String> outputs = gameManager.getPlayer().getOutputs();

            // If the output is not correct, the player loose the game (the messages are in the checkOuput function)
            Integer value = this.checkOutput(outputs);
            if(value == null)
                return;

            // Add the value to the BST. An exception may occurs, in which case the player looses the game
            binarySearchTree.addNode(value);

            // Update the GUI
            int x = binarySearchTree.getLastX();
            int y = binarySearchTree.getLastY();
            int fatherX = binarySearchTree.getLastFatherX();
            int fatherY = binarySearchTree.getLastFatherY();
            guiManager.place(value, x, y, fatherX, fatherY);

            // Check if all the goals are reached
            if(binarySearchTree.allGoalsReached()){
                gameManager.winGame("Success!");
                return;
            }


        } catch (AbstractPlayer.TimeoutException e) {
            // This exception occurs if the player is too slow.
            gameManager.loseGame("Timeout!");
            return;
        } catch (BombException | SameCoordinatesException | OutOfBoundException | SameIntegerException e){
            // This exception occurs if the player do a forbidden action : touch a bomb, add the same integer twice,
            // add an integer out of the bounds of the tree or put an integer at the same coordinates than another
            // integer.
            gameManager.loseGame(e.getMessage());
            return;
        }

        // If all the integer were placed and not all the goals are reached, the player loose.
        if(binarySearchTree.size() == n)
            gameManager.loseGame("Some goals were not reached!");


    }

    /**
     * @param outputs : Output of the player
     * @return an integer corresponding to the command of the player or null if the command is incorrect.
     */
    private Integer checkOutput(List<String> outputs) {

        // Check only one line was outputted.
        if (outputs.size() != 1) {
            gameManager.loseGame("You did not send 1 output in your turn.");
            return null;
        }

        // Check if the line contains one positive integer
        String output = outputs.get(0);
        if(!output.matches("\\d+")){
            gameManager.loseGame("You did not send an integer between 1 and n.");
            return null;
        }

        Integer value;
        // Check if the value is between 1 and n
        try{
            value = Integer.valueOf(output);
            if(value < 1 || value > n){
                gameManager.loseGame("You did not send an integer between 1 and n.");
                return null;
            }
        }
        catch(NumberFormatException e){
            gameManager.loseGame("You did not send an integer between 1 and n.");
            return null;
        }

        return value;

    }

    @Override
    public void onEnd() {
    }
}
