package com.codingame.gui;

import com.codingame.gameengine.module.entities.*;

/**
 * Graphical interface for the binary tree search game.
 *
 */
public class GuiManager {

    /**
     * Drawer of the graphical interface
     */
    private GraphicEntityModule graphicEntityModule;

    /**
     * Margin between two nodes of the tree.
     */
    private int margin;

    /**
     * Width of the tree
     */
    private int width;

    /**
     * Height of the tree
     */
    private int height;

    /**
     * Height of each cell. A cell is a rectangle containing a node of the tree.
     */
    private int cellsHeight;

    /**
     * Width of each cell. A cell is a rectangle containing a node of the tree.
     */
    private int cellsWidth;

    /**
     * All the circles. Each circle may contain a node of the tree, a bomb or a goal.
     */
    private Circle[][] circles;

    /**
     * All the lines. Each line connects two circles of {@link #circles}.
     */
    private Line[][][] lines;

    /**
     * Build a new GUI for the game, with the given graphic entity module, the given width and the given height.
     * @param graphicEntityModule
     * @param width
     * @param height
     */
    public GuiManager(GraphicEntityModule graphicEntityModule, int width, int height) {
        this.width = width;
        this.height = height;
        this.graphicEntityModule = graphicEntityModule;

        // Set the graphical parameters according to the width and the height of the world.

        this.margin = Math.min(Parameters.MARGIN, (int)(0.2 * (World.DEFAULT_HEIGHT / height)));

        this.cellsHeight =
                Math.min((World.DEFAULT_HEIGHT - margin) / height,
                        Parameters.MAX_CELL_HEIGHT + margin) - margin;

        this.cellsWidth =
                Math.min(((World.DEFAULT_WIDTH - margin) / width),
                        Parameters.MAX_CELL_WIDTH + margin) - margin;
    }

    /**
     * Init the GUI at the beginning of the game.
     *
     * Draw the background, all the circles in which an integer, a bomb or a goal can be contained, and all the
     * lines connecting those circles.
     */
    public void initGui(){
        // Draw background
        graphicEntityModule.createSprite()
                .setImage(Parameters.BACKGROUND_SPRITE)
                .setBaseWidth(World.DEFAULT_WIDTH)
                .setBaseHeight(World.DEFAULT_HEIGHT);

        // Draw the circles and the lines
        circles = new Circle[width][height];
        lines = new Line[width][height][2];

        for(int x = 0; x < width; x ++){
            for(int y = 0; y < height; y++){
                // Not all the coordinates contain a circle.
                if(width/2 % 2 == 1 && y % 2 == 0 && x % 2 == 0)
                    continue;
                if(width/2 % 2 == 1 && y % 2 == 1 && x % 2 == 1)
                    continue;
                if(width/2 % 2 == 0 && y % 2 == 1 && x % 2 == 0)
                    continue;
                if(width/2 % 2 == 0 && y % 2 == 0 && x % 2 == 1)
                    continue;
                addCircle(x, y);
            }
        }
    }

    /**
     * Draw a circle at the given coordinates. If the circle is the root, the color of the circle is drawn blue-ish.
     * To that circle are added two lines connecting this circle to the upper left circle and the upper right circle.
     * @param x
     * @param y
     */
    private void addCircle(int x, int y){
        Circle circle = graphicEntityModule.createCircle()
                .setX(getX(x) + cellsWidth / 2)
                .setY(getY(y) - cellsHeight / 2)
                .setRadius(cellsHeight / 2)
                .setLineWidth(5);

        // Check if the circle is the root circle
        if(x == width / 2 && y == 0)
            circle.setLineColor(0x000000).setFillColor(0x77cccc);
        else if(Math.abs(x - width / 2) > y)
            circle.setLineColor(0x000000).setFillColor(0x555555);
        else
            circle.setLineColor(0x000000).setFillColor(0xeeeeee);

        circles[x][y] = circle;

        // Add the lines (except if the circle is on the upper line of the grid)
        if(y != height - 1){
            if(x != 0)
                lines[x][y][0] = graphicEntityModule.createLine()
                    .setX((int)(getX(x) + cellsWidth / 2 - (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setY((int)(getY(y) - cellsHeight / 2 + (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setX2((int)(getX(x - 1) + cellsWidth / 2 + (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setY2((int)(getY(y + 1) - cellsHeight / 2 - (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setLineWidth(5)
                    .setLineColor(0x000000);

            if(x != width - 1)
                lines[x][y][1] = graphicEntityModule.createLine()
                    .setX((int)(getX(x) + cellsWidth / 2 + (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setY((int)(getY(y) - cellsHeight / 2 + (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setX2((int)(getX(x + 1) + cellsWidth / 2 - (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setY2((int)(getY(y + 1) - cellsHeight / 2 - (cellsHeight / 2) * (Math.sqrt(2) / 2)))
                    .setLineWidth(5)
                    .setLineColor(0x000000);
        }
    }

    /**
     * Add a bomb at the given coordinates
     * @param x
     * @param y
     */
    public void addBomb(int x, int y){
        graphicEntityModule.createSprite()
                .setImage(Parameters.BOMB_SPRITE)
                .setX(getX(x) + cellsWidth / 2 - (int)(cellsHeight * 0.9) / 2)
                .setY(getY(y) - cellsHeight + 10)
                .setBaseWidth((int)(cellsHeight * 0.9))
                .setBaseHeight((int)(cellsHeight * 0.9));

    }

    /**
     * Add a goal at the given coordinates
     * @param x
     * @param y
     */
    public void addGoal(int x, int y){
        graphicEntityModule.createSprite()
                .setImage(Parameters.GOAL_SPRITE)
                .setX(getX(x) + cellsWidth / 2 - (int)(cellsHeight * 0.9) / 2)
                .setY(getY(y) - cellsHeight)
                .setBaseWidth((int)(cellsHeight * 0.9))
                .setBaseHeight((int)(cellsHeight * 0.9));
    }

    /**
     * @param x
     * @return the x-pixel-coordinate cooresponding to the given x-coordinate
     */
    private int getX(int x){
        return World.DEFAULT_WIDTH / 2 - cellsWidth / 2 +  (x - width / 2) * (cellsWidth + margin);
    }

    /**
     * @param y
     * @return the y-pixel-coordinate cooresponding to the given y-coordinate
     */
    private int getY(int y){
        return (y  + 1) * (margin + cellsHeight);
    }

    /**
     * Place the given value at the given (x, y) coordinates, and link it to the node place at the given (fatherX,
     * fatherY) coordinates.
     * @param value
     * @param x
     * @param y
     * @param fatherX
     * @param fatherY
     */
    public void place(int value, int x, int y, int fatherX, int fatherY){

        Text valueText = graphicEntityModule.createText(String.valueOf(value))
                .setX(getX(x) + cellsWidth / 2 - 15 * (int)(Math.log10(value) + 1))
                .setY(getY(y) - cellsHeight / 2 - 25)
                .setFontFamily("monospace")
                .setFontSize(50)
                .setFillColor(0x000000)
                .setAlpha(0);


        graphicEntityModule.commitWorldState(0.1);

        valueText.setAlpha(1);
        circles[x][y].setFillColor(0xffaaaa);
        if(fatherX != -1 && fatherY != -1)
            lines[fatherX][fatherY][(fatherX > x)?0:1].setLineWidth(15);
        graphicEntityModule.commitWorldState(0.9);

    }

}
