package com.codingame.game.heap;

/**
 * A node of a binary search tree.
 * Such a node contains a integer value, a couple of (x, y)-coordinates, is possibly linked to a father, a left child
 * and a right child.
 *
 * The values of the nodes in the subtree rooted at the left children are lower than the value of this node, and the
 * values of the right subtree are greater.
 */
class BinarySearchTreeNode {

    /**
     * The x-coordinate of this node.
     */
    private int x;

    /**
     * The y-coordinate of this node.
     */
    private int y;

    /**
     * The value of this node.
     */
    private int value;

    /**
     * The father of this node, if it exists.
     */
    private BinarySearchTreeNode father;

    /**
     * The left child of thie node, if it exists.
     */
    private BinarySearchTreeNode left;

    /**
     * The right child of thie node, if it exists.
     */
    private BinarySearchTreeNode right;

    /**
     * Create a new BST node with the given father (null if this node is the root), placed at coordinates (x, y) and
     * associated with the given value.
     * @param father
     * @param x
     * @param y
     * @param value
     */
    BinarySearchTreeNode(BinarySearchTreeNode father, int x, int y, int value) {
        this.father = father;
        this.x = x;
        this.y = y;
        this.value = value;
    }

    /**
     * @return the father of this node, or null if it does not exist.
     */
    BinarySearchTreeNode getFather() {
        return father;
    }

    /**
     * @return the x-coordinate of this node.
     */
    int getX() {
        return x;
    }

    /**
     * @return the y-coordinate of this node.
     */
    int getY() {
        return y;
    }

    /**
     * @return the value associated with this node
     */
    int getValue() {
        return value;
    }

    /**
     * Add the value of the subtree rooted at this node if no node in that tree already contains that value.
     *
     * @param value
     * @return the new created descendant node or null if a descendant of this node already contained the given value.
     */
    BinarySearchTreeNode addChild(int value){
        if(this.getValue() == value)
            return null;
        else if(this.getValue() > value)
            return this.addLeft(value);
        else
            return this.addRight(value);
    }

    /**
     * Add the value of the subtree rooted at the left child of this if no node in that tree already contains that
     * value. If no left child exists, create a new left child with the given value.
     *
     * @param value
     * @return the new created descendant node or null if a descendant of the left child already contained the given
     * value.
     */
    private BinarySearchTreeNode addLeft(int value) {
        if(this.left == null){
            BinarySearchTreeNode node = new BinarySearchTreeNode(this,this.x - 1, this.y + 1, value);
            this.left = node;
            return node;
        }
        return left.addChild(value);
    }

    /**
     * Add the value of the subtree rooted at the right child of this if no node in that tree already contains that
     * value. If no right child exists, create a new right child with the given value.
     *
     * @param value
     * @return the new created descendant node or null if a descendant of the right child already contained the given
     * value.
     */
    private BinarySearchTreeNode addRight(int value) {
        if(this.right == null){
            BinarySearchTreeNode node = new BinarySearchTreeNode(this, this.x + 1, this.y + 1, value);
            this.right = node;
            return node;
        }
        return right.addChild(value);
    }
}
