package com.codingame.game.heap;

import com.codingame.game.exceptions.BombException;
import com.codingame.game.exceptions.OutOfBoundException;
import com.codingame.game.exceptions.SameCoordinatesException;
import com.codingame.game.exceptions.SameIntegerException;

import java.util.LinkedList;

/**
 * Model of a binary search tree, with bombs and goals.
 * Integer values can be added to the tree with the method {@link #addNode(int)}.
 *
 * The tree has an (odd) {@link #width} and a {@link #height}. Each node has coordinates that must belongs to
 * [0; width - 1] x [0 ; height - 1]. The root is placed at coordinates (({@link #width} - 1) / 2, 0). The left child of a node
 * at coordinates (x, y) is placed at position (x - 1, y + 1), and the right chikd is placed at (x + 1, y + 1).
 *
 * As in every binary search tree, every descendant of the left child of a node of value V has a lower value whereas
 * the other nodes have a value greater than V.
 *
 * To the tree are associated bombs and goals. Each bomb/goal is placed at some coordinates. When an integer is added
 * to the tree, it should not be placed where a bomb is located. On the contrary, all the goals must be touched by the
 * tree.
 *
 * Remark : the height of the tree is always the maximum y-coordinate of the goals plus one.
 */
public class BinarySearchTree {

    /**
     * Width of the tree, the x-coordinates of the nodes should be between 0 and {@link #width} - 1. The root is
     * always at coordinates (({@link #width} - 1) / 2, 0).
     */
    private int width;

    /**
     * Height of the tree, the y-coordinates of the nodes should be between 0 and {@link #height} - 1.
     * The height is always the maximum y-coordinate of the goals plus one.
     */
    private int height;

    /**
     * List of nodes currently in the tree.
     */
    private LinkedList<BinarySearchTreeNode> nodes;

    /**
     * List of bombs associated with the tree. bombs[i] contains a couple of integers (x, y) corresponding to the
     * coordinates of the i-th bomb.
     */
    private int[][] bombs;

    /**
     * List of goals associated with the tree. goals[i] contains a couple of integers (x, y) corresponding to the
     * coordinates of the i-th goal.
     */
    private int[][] goals;

    /**
     * Return the number of goals currently reached by the tree.
     */
    private int goalsReached;

    /**
     * Build a binary tree with the given width, with no bomb and no goal.
     * The height of the tree is 0 as it contains no goal and no node.
     *
     * @param width : width of the tree, the x-coordinates of the nodes should be between 0 and {@link #width} - 1.
     */
    public BinarySearchTree(int width) {
        this.width = width;
        this.height = 0;
        nodes = new LinkedList<>();
    }

    /**
     * Set the bombs of the tree.
     *
     * @param bombs : Description of the bombs. bombs[i] should contain a couple of integers (x, y) corresponding to
     * the coordinates of the i-th bomb.
     */
    public void setBombs(int[][] bombs){
        this.bombs = bombs;
    }

    /**
     * Set the goals of the tree, and update the height of the tree.
     *
     * @param goals : Description of the goals. goals[i] should contain a couple of integers (x, y) corresponding to
     * the coordinates of the i-th goal.
     * @see #height
     */
    public void setGoals(int[][] goals){
        this.goals = goals;
        for(int[] goal : goals)
            this.height = Math.max(this.height, goal[1] + 1);
    }

    /**
     * Add the given value to the tree.
     *
     * @param value
     * @throws OutOfBoundException : if the value is added to the tree with a node placed at coordinates (x, y) such
     * that x < 0, x >= width, y < 0 or y >= height.
     * @throws BombException : if the value is added to the tree with a node placed at the coordinates of some bomb.
     * @throws SameCoordinatesException : if the value is added to the tree with a node placed at the coordinates of
     * some other existing node.
     * @throws SameIntegerException : if the value is already contained in the tree
     */
    public void addNode(int value) throws OutOfBoundException, BombException, SameCoordinatesException, SameIntegerException {

        // Node that wille be created to contain the given value.
        BinarySearchTreeNode node;
        if(nodes.size() == 0) // The node should be the root
            node = new BinarySearchTreeNode(null,width / 2, 0, value);
        else // The node should be added as a left/right child of an existing node
            node = nodes.getFirst().addChild(value);

        // The value is already in the tree, no node was added.
        if(node == null)
            throw new SameIntegerException(value);

        int x = node.getX();
        int y = node.getY();

        // Check the x and y coordinates compared to the bounds of the tree
        if(x < 0 || x >= width)
            throw new OutOfBoundException(value, x, y);

        if(y < 0 || y >= height)
            throw new OutOfBoundException(value, x, y);

        // Check is no bomb is touched
        for (int[] bomb : bombs)
            if (bomb[0] == x && bomb[1] == y)
                throw new BombException(value, x, y);

        // Check if no existing node is placed at the same coordinates than the new created node
        for(BinarySearchTreeNode node2: nodes)
            if(node2.getX() == x && node2.getY() == y)
                throw new SameCoordinatesException(value, node2.getValue(), x, y);

        // Add the node to the list of nodes
        nodes.add(node);

        // Check if some goal is reached.
        for (int[] goal : goals)
            if (goal[0] == x && goal[1] == y) {
                goalsReached++;
                break;
            }
    }


    /**
     * @return true if all the goals if the tree are reached.
     */
    public boolean allGoalsReached() {
        return goalsReached == goals.length;
    }

    /**
     * @return the x-coordinate of the node containing the last added value.
     */
    public int getLastX() {
        return nodes.getLast().getX();
    }

    /**
     * @return the y-coordinate of the node containing the last added value.
     */
    public int getLastY() {
        return nodes.getLast().getY();
    }

    /**
     * @return the x-coordinate of the father of the node containing the last added value or -1 if that node is the
     * root.
     */
    public int getLastFatherX() {
        BinarySearchTreeNode father = nodes.getLast().getFather();
        if(father == null)
            return -1;
        return father.getX();

    }

    /**
     * @return the y-coordinate of the father of the node containing the last added value if that node is the
     * root.
     */
    public int getLastFatherY() {
        BinarySearchTreeNode father = nodes.getLast().getFather();
        if(father == null)
            return -1;
        return father.getY();
    }

    /**
     * @return the number of nodes in the tree.
     */
    public int size(){
        return nodes.size();
    }
}
