package com.codingame.game;

import com.codingame.gameengine.module.entities.*;
import com.google.inject.Inject;
import re.desast.freecell.*;
import re.desast.freecell.Card.Suit;
import re.desast.freecell.Card.Rank;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;

class View {
    @Inject private GraphicEntityModule gem;
    private String[] cards;

    private static final int TITLE_X = 294;
    private static final int TITLE_Y = 13;

    private static final int CARD_LEFT_X = 1662;
    private static final int CARD_LEFT_Y = 59;

    private BitmapText cardsLeft;

    private static final int HEAD_X = 920;
    private static final int HEAD_Y = 138;

    private final EnumMap<Side,Sprite> heads = new EnumMap<>(Side.class);

    void init(int gameNo) {
        cards = gem.createSpriteSheetSplitter()
                .setSourceImage("Cards.svg")
                .setImageCount(52)
                .setWidth(390)
                .setHeight(570)
                .setOrigRow(0)
                .setOrigCol(0)
                .setImagesPerRow(13)
                .setName("card")
                .split();

        gem.createSprite().setImage("FreeCell.png").setZIndex(-256);
        gem.createBitmapText().setFont("MS Sans Bold")
                .setText(" " + Integer.valueOf(gameNo).toString())
                .setX(TITLE_X)
                .setY(TITLE_Y)
                .setFontSize(20)
                .setZIndex(-255);

        cardsLeft = gem.createBitmapText().setFont("MS Sans Regular")
                .setX(CARD_LEFT_X)
                .setY(CARD_LEFT_Y)
                .setFontSize(20)
                .setAnchorX(1)
                .setZIndex(-255);
        setCardsLeft(52, 0, 1);

        heads.put(Side.LEFT,  gem.createSprite().setImage("King-left.png" ).setX(HEAD_X).setY(HEAD_Y).setVisible(false));
        heads.put(Side.RIGHT, gem.createSprite().setImage("King-right.png").setX(HEAD_X).setY(HEAD_Y).setVisible(false));
    }

    void setCardsLeft(int cards, int i, int n) {
        cardsLeft.setText(" " + Integer.valueOf(cards).toString());
        gem.commitEntityState((double) (i+1) / n, cardsLeft);
    }

    final Entity<?>[] deck = new Entity[52];

    void createCard(int index, Suit suit, Rank rank) {
        deck[index] = gem.createSprite()
                .setImage(cards[13*(3 - suit.ordinal()) + rank.ordinal() - 1])
                .setScale(160.0/364);
    }

    void createCardsFromCascades(re.desast.freecell.Cascade[] cascades) {
        Card[][] piles = Arrays.stream(cascades).map(c -> c.stream().toArray(Card[]::new)).toArray(Card[][]::new);
        for (int i = 0; i < 7; i++) {
            for (int j = 0; j < piles.length; j++) {
                if (i >= piles[j].length) continue;
                Card card = piles[j][i];
                createCard(card.index, card.suit, card.rank);
                moveCard(card.index, Cascade.getX(cascades[j].slotName()), Cascade.getY(i), Cascade.getZ(i), 0, 1);
            }
        }
    }

    public void moveCard(int index, int x, int y, int z, int i, int n) {
        Entity<? extends Entity<?>> card = deck[index];
        card.setZIndex(256 + z);
        gem.commitEntityState((double) i / n, card);
        card.setX(x).setY(y);
        gem.commitEntityState((double) (i+1) / n - 0.000001, card); // epsilon = GEM serializer formatFrameTime (undocumented) resolution
        card.setZIndex(z);
        gem.commitEntityState((double) (i+1) / n, card);
    }

    void viewCardsMove(Move move, int i, int n, List<Card> cards) {
        int destX, destY, destZ, spacerY;
        if (move.dst instanceof re.desast.freecell.Cascade) {
            destX = Cascade.getX(move.dst.slotName());
            destY = Cascade.getY(((re.desast.freecell.Cascade) move.dst).getHeight());
            destZ = Cascade.getZ(((re.desast.freecell.Cascade) move.dst).getHeight());
            spacerY = Cascade.Y_SPACE;
        }
        else if (move.dst instanceof re.desast.freecell.Cell) {
            destX = Cell.getX(move.dst.slotName());
            destY = Cell.getY();
            destZ = Cell.getZ();
            spacerY = 0;
        }
        else if (move.dst instanceof re.desast.freecell.Foundation) {
            destX = Foundation.getX(((re.desast.freecell.Foundation) move.dst).suit);
            destY = Foundation.getY();
            destZ = Foundation.getZ(((re.desast.freecell.Foundation) move.dst).contents.size());
            spacerY = 0;
        }
        else throw new RuntimeException("Unexpected move destination type for " + move.dst.slotName() + ": " + move.dst);
        int z = move.count;
        for (Card card : cards) moveCard(card.index, destX, destY + --z * spacerY, destZ + z, i, n);
    }

    enum Side { LEFT, RIGHT }

    private void setHeadSide(Side side) {
        for (Side s : Side.values()) gem.commitEntityState(0, heads.get(s).setVisible(s == side));
    }

    void adjustHead(ReceiverSlot dst) {
        Side target;
        if (dst instanceof re.desast.freecell.Cell
            || dst instanceof re.desast.freecell.Cascade && ((re.desast.freecell.Cascade) dst).index < 4)
            target = Side.LEFT;
        else if (dst instanceof re.desast.freecell.Foundation
                 || dst instanceof re.desast.freecell.Cascade && ((re.desast.freecell.Cascade) dst).index >= 4)
            target = Side.RIGHT;
        else throw new Exception("Unexpected move destination type for " + dst.slotName() + ": " + dst);

        setHeadSide(target);
    }

    private static final int CARD_WIDTH = 160;
    private static final int CARD_HEIGHT = 239;

    private static final int PANE_X = 249;
    private static final int PANE_Y = 95;
    private static final int PANE_WIDTH = 1422;

    static class Cascade {
        private static final int X_REMAINDER = PANE_WIDTH - 8 * CARD_WIDTH;
        private static final int Y_SPACE = 50;
        private static int slotIndex(char slot) {
            if (slot < '1' || slot > '8') throw new Exception("Invalid cascade slot: " + slot);
            else return slot - '1';
        }
        static int getX(char slot) { return PANE_X + ((X_REMAINDER + 9 * CARD_WIDTH) * slotIndex(slot) + X_REMAINDER + 4) / 9; }
        static int getY(int h) { return PANE_Y + CARD_HEIGHT + X_REMAINDER/9 + Y_SPACE * h; }
        static int getZ(int h) { return h; }
    }

    static class Cell {
        private static int slotIndex(char slot) {
            if (slot < 'a' || slot > 'd') throw new Exception("Invalid cell slot: " + slot);
            else return slot - 'a';
        }
        static int getX(char slot) { return PANE_X + CARD_WIDTH * slotIndex(slot); }
        static int getY() { return PANE_Y; }
        static int getZ() { return 0; }
    }

    static class Foundation {
        private static int suitIndex(Suit suit) { return suit.ordinal(); }
        static int getX(Suit suit) { return PANE_X + PANE_WIDTH + CARD_WIDTH * (suitIndex(suit) - 4); }
        static int getY() { return PANE_Y; }
        static int getZ(int i) { return i; }
    }

    private static final int VICTORY_X = 249;
    private static final int VICTORY_Y = 334;

    public void victory() {
        gem.commitEntityState(0, gem.createSprite().setImage("Victory.png").setX(VICTORY_X).setY(VICTORY_Y));
    }

    private static class Exception extends RuntimeException {
        Exception(String msg) { super (msg); }
    }
}
