package re.desast.freecell;

import java.util.*;
import java.util.stream.Stream;

/**
 * A tableau is a sequence of cards of one-by-one decreasing ranks and of alternating colors.
 *
 * We enforce the invariant that tableaux are never empty.
 */
public final class Tableau implements CardProvider, CardReceiver, Cards {
    private final LinkedList<Card> deque;

    Tableau(Card card) {
        deque = new LinkedList<>();
        deque.add(card);
    }

    private Tableau(LinkedList<Card> deque) {
        this.deque = deque;
    }

    @Override public Tableau toTableau() { return this; }

    @Override public Optional<Card> toCard() {
        if (deque.size() == 1) return Optional.of(deque.element());
        else return Optional.empty();
    }

    @Override public Optional<Card> proposeCard() {
        return Optional.of(deque.getLast());
    }

    @Override public Optional<Cards> proposeCards(int limit) {
        return Optional.of(truncated(limit));
    }

    @Override
    public Cards removeCards(int count) {
        if (count >= deque.size()) throw new IllegalArgumentException("Emptying tableau");
        if (count < 1) throw new IllegalArgumentException("Nonsense NOP request");

        LinkedList<Card> cards = new LinkedList<>();
        while (count --> 0) cards.addFirst(deque.removeLast());
        return new Tableau(cards);
    }

    @Override
    public Optional<Integer> wouldAcceptCards(Cards cards) {
        if (((getLowestRank().rank - cards.getLowest().rank.rank & 1) == 1) == (getLowestColour() == cards.getLowest().suit.colour)) return Optional.empty();
        int count = getLowestRank().rank - cards.getLowest().rank.rank;
        if (count <= 0 || count > cards.size()) return Optional.empty();
        else return Optional.of(count);
    }

    @Override
    public void acceptCards(Cards cards) {
        if (wouldAcceptCards(cards).isPresent()) for (Card card : cards.toTableau().deque) deque.addLast(card);
        else throw new RuntimeException("Internal error: invalid move");
    }

    @Override
    public Cards truncated(int limit) {
        if (limit >= size()) return this;
        return new Tableau(new LinkedList<>(deque.subList(deque.size() - limit, deque.size())));
    }

    @Override
    public Card getLowest() {
        return deque.getLast();
    }

    public Card.Rank getLowestRank() {
        return deque.getLast().rank;
    }

    public Card.Colour getLowestColour() {
        return deque.getLast().suit.colour;
    }

    public int size() { return deque.size(); }

    public Stream<Card> stream() {
        return deque.stream();
    }

    @Override
    public Iterator<Card> iterator() {
        return deque.descendingIterator();
    }
}
