package re.desast.freecell;

import org.jetbrains.annotations.NotNull;

import java.util.*;

public class Card implements Cards, Comparable<Card> {

    public static Queue<Card> newDeck(int seed) {
        Card[] deck = new Card[52];
        for (int i = 0; i < 52; i++) {
            deck[i] = new Card(i, 1 + i / 4, Suit.getSuit(i % 4));
        }

        // this is Fisher-Yates, but it's dovetail with card generation above and RNG usage to strictly reproduce the MS deals.
        RNG rng = new RNG(seed);
        Queue<Card> output = new ArrayDeque<>(52);
        for (int wLeft = 52; wLeft > 0; wLeft--) {
            int j = rng.next() % wLeft;
            output.add(deck[j]);
            deck[j] = deck[wLeft-1];
        }
        return output;
    }

    public final int index;
    public final Rank rank;
    public final Suit suit;

    public String toIoString() { return rank.io + suit.ascii; }

    private Card(int index, int rank, Suit suit) {
        this.index = index;
        this.rank = Rank.get[rank];
        this.suit = suit;
    }

    @Override public int size() { return 1; }
    @Override public Tableau toTableau() { return new Tableau(this); }
    @Override public Optional<Card> toCard() { return Optional.of(this); }
    @Override public Cards truncated(int limit) {
        throw new IllegalArgumentException("Truncating a card");
    }
    @Override public Card getLowest() { return this; }

    @Override @NotNull
    public Iterator<Card> iterator() {
        return Collections.singleton(this).iterator();
    }

    @Override
    public int compareTo(@NotNull Card card) {
        return Comparator.comparing((c -> ((Card) c).rank)).thenComparing(c -> ((Card) c).suit).compare(this, card);
    }

    public enum Colour { BLACK, RED }

    public enum Suit {
        CLUBS   (0, Colour.BLACK, "C", "♣", 0x000000),
        DIAMONDS(1, Colour.RED,   "D", "♦", 0xFF0000),
        HEARTS  (2, Colour.RED,   "H", "♥", 0xFF0000),
        SPADES  (3, Colour.BLACK, "S", "♠", 0x000000);

        public final int index;
        public final Colour colour;
        public final String ascii;
        public final String unicode;
        public final int color; // TODO: suit color is view-related

        Suit(int index, Colour colour, String ascii, String unicode, int color) {
            this.index = index;
            this.colour = colour;
            this.ascii = ascii;
            this.unicode = unicode;
            this.color = color;
        }

        static final Suit[] suits = { CLUBS, DIAMONDS, HEARTS, SPADES };
        static Suit getSuit(int index) { return suits[index]; }
    }

    public enum Rank {
        NONE ( 0, '0', null), // a sentry for foundations TODO use as such
        ACE  ( 1, 'A', "A"),
        TWO  ( 2, '2', "2"),
        THREE( 3, '3', "3"),
        FOUR ( 4, '4', "4"),
        FIVE ( 5, '5', "5"),
        SIX  ( 6, '6', "6"),
        SEVEN( 7, '7', "7"),
        EIGHT( 8, '8', "8"),
        NINE ( 9, '9', "9"),
        TEN  (10, 'T', "10"),
        JACK (11, 'J', "J"),
        QUEEN(12, 'Q', "Q"),
        KING (13, 'K', "K");

        public final int rank;
        public final char io;
        public final String display;

        Rank(int rank, char io, String display) {
            this.rank = rank;
            this.io = io;
            this.display = display;
        }

        private final static Rank[] get = {
            NONE, ACE, TWO, THREE, FOUR, FIVE, SIX, SEVEN, EIGHT, NINE, TEN, JACK, QUEEN, KING
        };
    }
}
