package re.desast.freecell;

import re.desast.util.PeekableIterator;
import re.desast.util.PeekableIteratorAdaptor;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class Game {
    public static int randomMsWinnableSeed() {
        Random r = new Random();
        while (true) {
            int seed = 1 + r.nextInt() % 32000;
            if (seed != 11982) return seed;
        }
    }
    public final Foundation[] foundations = new Foundation[4];
    public final Cell[] cells = new Cell[4];
    public final Cascade[] cascades = new Cascade[8];
    public final ProviderSlot[] providers = new ProviderSlot[12];
    public final ReceiverSlot[] receivers = new ReceiverSlot[16];

    public Game(int seed) {
        // slots
        for (int i = 0; i < 4; i++) {
            cells[i] = new Cell(i);
            providers[i] = cells[i];
            receivers[i] = cells[i];
        }
        for (int i = 0; i < 4; i++) {
            foundations[i] = new Foundation(i);
            receivers[i+4] = foundations[i];
        }
        for (int i = 0; i < 8; i++) {
            cascades[i] = new Cascade(i);
            providers[i+4] = cascades[i];
            receivers[i+8] = cascades[i];
        }

        // cards
        Queue<Card> deck = Card.newDeck(seed);
        int i = 0;
        for (Card c : deck) cascades[i++ % 8].deal(c);
    }

    public boolean gameWon() {
        return Arrays.stream(foundations).allMatch(f -> f.contents.size() == 13);
    }

    public Move moveFromIo(String io) throws BadMoveException {
        if (io.length() != 2) throw new BadMoveException("Wrong length word for a move: `" + io + "'");

        char inputIo = io.charAt(0);
        CardProvider src = providerFromIo(inputIo);
        if (!src.proposeCard().isPresent()) throw new BadMoveException("Can't move from empty slot " + inputIo + "!");

        Move move = moves.get(io);
        if (move == null) throw new BadMoveException("Illegal move: `" + io + "' !");
        return move;
    }

    public Optional<CardProvider> slotFromIo(char io) throws BadMoveException {
        if (Character.isLetter(io)) {
            io = Character.toLowerCase(io);
            if (io == 'h') return Optional.empty();
            if (io >= 'a' && io <= 'd') return Optional.of(cells[io - 'a']);
        }
        else if (io >= '1' && io <= '8') return Optional.of(cascades[io - '1']);
        throw new BadMoveException("Unknown slot: `" + io + "'");
    }

    public CardProvider providerFromIo(char io) throws BadMoveException {
        return slotFromIo(io).orElseThrow(() -> new BadMoveException("Can't move from foundations!"));
    }

    public List<String> dump() {
        List<String> result = new ArrayList<>();

        {
            StringBuilder buf = new StringBuilder();
            buf.append("Foundations:");
            for (Foundation f : foundations) {
                if (f.contents.size() == 0) continue;
                buf.append(' ');
                buf.append(f.suit.ascii);
                buf.append('-');
                if (f.contents.empty()) buf.append('0');
                else buf.append(f.contents.peek().rank.io);
            }
            result.add(buf.toString());
        }

        {
            StringBuilder buf = new StringBuilder();
            buf.append("Freecells:");
            for (Cell c : cells) {
                buf.append(' ');
                if (c.contents == null) buf.append('-');
                else buf.append(c.contents.toIoString());
            }
            result.add(buf.toString());
        }

        for (Cascade c : cascades) {
            result.add(Stream.concat(Stream.of(":"),c.stream().map(Card::toIoString)).collect(Collectors.joining(" ")));
        }

        return result;
    }

    private final Map<String,Move> moves = new TreeMap<>(); // keeping them ordered
    private void putMove(Move m) { moves.put(m.toString(), m); }

    public Collection<Move> generateMoves() {
        moves.clear();

        int freeCells = (int) Arrays.stream(cells).filter(cell -> cell.contents == null).count();
        int freeCascades = (int) Arrays.stream(cascades).filter(cascade -> cascade.contents.empty()).count();
        int mobility = (freeCells + 1) * (1 << freeCascades);
        int reducedMobility = mobility / 2;

        for (ProviderSlot src : providers) {
            Optional<Cards> optCards = src.proposeCards(mobility);
            if (!optCards.isPresent()) continue;

            Cards moreCards = optCards.get();
            Cards fewerCards = moreCards.size() > 1 ? moreCards.truncated(reducedMobility) : moreCards;

            for (ReceiverSlot dst : receivers) {
                if (dst == src) continue;

                Cards cards = dst.isEmptyCascade() ? fewerCards : moreCards;
                dst.wouldAcceptCards(cards).ifPresent(count -> putMove(new Move(ts, count, src, dst)));
            }
        }
        return moves.values();
    }

    public List<Move> generateAutoplay() {
        List<Move> moves = new ArrayList<>();
        TimeStamp date = ts;
        NavigableMap<PeekableIterator<Card>, ProviderSlot> available =
            new TreeMap<>(
                Comparator.comparing(PeekableIterator::peek, Comparator.nullsLast(Comparator.naturalOrder()))
            );
        for (ProviderSlot provider : providers) available.put(new PeekableIteratorAdaptor<>(provider.iterator()), provider);

        // failed to just use streams here :-( something to do with generic array creation being tricky
        Map<Card.Suit,Card.Rank> levels = new EnumMap<>(Card.Suit.class);
        for (Foundation f : foundations) levels.put(f.suit, Card.Rank.values()[f.contents.size()]);

        for (boolean active = true; active; ) {
            active = false;

            for (Map.Entry<PeekableIterator<Card>, ProviderSlot> entry : available.entrySet()) {
                Card card = entry.getKey().peek();
                if (card == null) break;
                if (levels.get(card.suit).rank == card.rank.rank - 1 && autoremovable(card, levels)) {
                    levels.replace(card.suit,card.rank);
                    moves.add(new Move(date, 1, entry.getValue(), foundations[card.suit.index]));
                    date = date.next();
                    available.remove(entry.getKey());
                    entry.getKey().next();
                    available.put(entry.getKey(), entry.getValue());
                    active = true;
                    break;
                }
            }
        }
        return moves;
    }

    private boolean autoremovable(Card card, Map<Card.Suit,Card.Rank> levels) {
        // MS rule
        return card.rank == Card.Rank.ACE || card.rank == Card.Rank.TWO ||
                levels.entrySet().stream()
                        .filter(e -> e.getKey().colour != card.suit.colour)
                        .allMatch(e -> e.getValue().rank >= card.rank.rank - 1);
    }

    public int getCardsLeft() {
        return 52 - Arrays.stream(foundations).map(f -> f.contents.size()).reduce(0, Integer::sum);
    }

    class TimeStamp {
        private TimeStamp _next;
        private TimeStamp() {}
        private TimeStamp next() { if (_next == null) _next = new TimeStamp(); return _next; }
        void tick() throws Exception {
            if (Game.this.ts != this) throw new Exception();
            Game.this.ts = next();
        }
        class Exception extends java.lang.Exception {}
    }
    private TimeStamp ts = new TimeStamp();
}
