package com.codingame.game;
import java.util.*;
import java.util.stream.Collectors;

import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.SoloGameManager;
import com.google.inject.Inject;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import re.desast.freecell.*;

public class Referee extends AbstractReferee {
    @Inject private SoloGameManager<Player> gm;
    @Inject private View view;
    private static final Log log = LogFactory.getLog(Referee.class);

    private Game game;
    private final Queue<String> actionQueue = new ArrayDeque<>();

    @Override
    public void init() {
        String testCaseInput = gm.getTestCaseInput().get(0);
        int seed = testCaseInput.equals("random") ? Game.randomMsWinnableSeed() : Integer.parseInt(testCaseInput);
        game = new Game(seed);

        view.init(seed);
        view.createCardsFromCascades(game.cascades);
    }

    private boolean needsAutoplay = false;

    @Override
    public void gameTurn(int turn) {
        try {
            if (game.gameWon()) vistory();
            else if (autoPlay());
            else humanPlay();
        }
        catch (FreeCellException e) {
            throw new RuntimeException(e);
        }
    }

    private void vistory() {
        gm.winGame("You won.  Congratulations!");
        view.victory();
        gm.setFrameDuration(1);
    }

    private void humanPlay() throws FreeCellException {
        log.info("Human play");

        Collection<Move> moves = game.generateMoves();

        if (actionQueue.isEmpty()) {
            Player player = gm.getPlayer();

            String foundationsCountIo = Long.toString(1 +
                    Arrays.stream(game.foundations)
                    .filter(f -> f.contents.size() > 0)
                    .count());
            String cascadeSizesIo = Arrays.stream(game.cascades)
                    .map(Cascade::getHeight)
                    .map(i -> Integer.toString(1 + i))
                    .collect(Collectors.joining(" "));
            List<String> dump = game.dump();

            if (moves.size() == 0) {
                gm.loseGame("No moves left.  You lose!");
                return;
            }

            player.sendInputLine(foundationsCountIo);
            player.sendInputLine(cascadeSizesIo);
            for (String l : dump) player.sendInputLine(l);
            player.sendInputLine(Integer.toString(moves.size()));
            player.sendInputLine(moves.stream().map(Move::toString).collect(Collectors.joining(" ")));

            player.execute();

            try {
                Collections.addAll(actionQueue, player.getOutputs().get(0).split(" "));
            }
            catch (TimeoutException e) {
                gm.loseGame("Timeout!");
                return;
            }
        }

        if (actionQueue.isEmpty()) {
            gm.loseGame("No action provided!");
            return;
        }

        log.info("Next move: " + actionQueue.peek());
        gm.setFrameDuration(1000);
        try { performAndViewMove(game.moveFromIo(actionQueue.remove()), 0, 1, true); }
        catch (BadMoveException e) {
            gm.loseGame(e.getMessage());
            return;
        }

        needsAutoplay = true;
    }

    private boolean autoPlay() throws FreeCellException {
        if (!needsAutoplay) return false;

        List<Move> moves = game.generateAutoplay();
        int n = moves.size();
        if (n > 0) {
            log.info("Autoplay");
            gm.setFrameDuration(200 * n);
            int i = 0;
            for (Move move : game.generateAutoplay()) {
                performAndViewMove(move, i++, n, false);
            }
        }
        return n > 0;
    }

    private void performAndViewMove(Move move, int i, int n, boolean adjustHead) throws FreeCellException {
        if (adjustHead) view.adjustHead(move.dst);

        List<Card> cards = move.performRetrieve();
        view.viewCardsMove(move, i, n, cards);
        move.performPlacement();

        view.setCardsLeft(game.getCardsLeft(), i, n);
    }

}
