package re.desast.freecell;

import re.desast.util.RecIterator;
import re.desast.util.StackIterator;

import java.util.*;
import java.util.stream.Stream;

public final class Cascade implements ProviderSlot, ReceiverSlot {
    public final int index;
    public final Stack<Tableau> contents;

    public Cascade(int index) {
        this.index = index;
        contents = new Stack<>();
    }

    public char slotName() { return (char) ('1' + index); }

    @Override
    public Optional<Card> proposeCard() {
        if (contents.empty()) return Optional.empty();
        else return contents.peek().proposeCard();
    }

    @Override
    public Cards removeCards(int count) {
        if (count <= 0) throw new IllegalArgumentException("Nonsensical remove of non-positive amount of cards");
        if (contents.empty()) throw new IllegalArgumentException("Nothing available to remove");

        Tableau available = contents.peek();
        if (count > available.size()) throw new IllegalArgumentException("Trying to remove more than can be at once");
        else if (count == available.size()) return contents.pop();
        else return available.removeCards(count);
    }

    @Override
    public Optional<Integer> wouldAcceptCards(Cards cards) {
        if (contents.empty()) return Optional.of(cards.size());
        else return contents.peek().wouldAcceptCards(cards);
    }

    @Override
    public void acceptCards(Cards cards) {
        if (contents.empty()) contents.push(cards.toTableau());
        else contents.peek().acceptCards(cards);
    }

    @Override
    public Optional<Cards> proposeCards(int limit) {
        if (contents.empty()) return Optional.empty();
        else return contents.peek().proposeCards(limit);
    }

    public int getHeight() {
        return (int) stream().count();
    }

    @Override
    public boolean isEmptyCascade() { return contents.empty(); }

    public void deal(Card card) {
        if (!contents.empty() && contents.peek().wouldAcceptCards(card).isPresent()) contents.peek().acceptCards(card);
        else contents.push(new Tableau(card));
    }

    public Stream<Card> stream() {
        return contents.stream().flatMap(Tableau::stream);
    }

    @Override
    public Iterator<Card> iterator() {
        return new RecIterator<>(new StackIterator<>(contents), Tableau::iterator);
    }

}
