package com.codingame.game;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.lang.Math;

import com.codingame.gameengine.core.AbstractMultiplayerPlayer;
import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.endscreen.EndScreenModule;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.entities.Sprite;
import com.codingame.gameengine.module.tooltip.TooltipModule;
import com.google.inject.Inject;

public class Referee extends AbstractReferee {
    private static int WIDTH = 1920;
    private static int HEIGHT = 1080;
    private static int BULLETSPEED = 60;
    private static int BULLETCOST = 2;
    private static int BULLETDMG = 20;
    private static int BULLETHEAL = 0;  // BULLETCOST*3;
    private static int EXPLODEPERSIST = 4;
    private static int MAXENERGY = 150;
    private static int MAXSPEED = 40;
    private static int HITBOX = 65;
    private static double MAXTURN = .3;
    private List<Sprite> bullets = new ArrayList<Sprite>();
    private List<Player> firedBy = new ArrayList<Player>();
    private List<Sprite> explosions = new ArrayList<Sprite>();
    private List<Integer> explosionCounter = new ArrayList<Integer>();

    @Inject private MultiplayerGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;
    @Inject private EndScreenModule endScreenModule;
    @Inject private TooltipModule tooltips;

    @Override
    public void init() {
        gameManager.setFrameDuration(150);
        int turns = 300;
        if(gameManager.getPlayerCount()==3)turns = 200;
        else if(gameManager.getPlayerCount()==4)turns = 150;
        gameManager.setMaxTurns(turns);
        gameManager.setTurnMaxTime(50);
        graphicEntityModule.createSprite().setImage("background.jpg").setAnchor(0);
        ArrayList<String> colors = new ArrayList<String>();
        colors.add("Red");colors.add("Blue");colors.add("Black");colors.add("Beige");

        for(Player p : gameManager.getPlayers()){
            p.energy = MAXENERGY;
            p.score = 0;
            String color = colors.get(0);
            colors.remove(color);

            // Assign each player a different color tank, and space them out across the middle of the board
            p.tank = graphicEntityModule.createSprite().setImage("tank"+color+".png").setAnchor(.5);
            p.turret = graphicEntityModule.createSprite().setImage("barrel"+color+".png").setAnchorX(.5).setAnchorY(.8);
            p.fullTank = graphicEntityModule.createGroup(p.tank,p.turret).setX((int)(WIDTH/(gameManager.getPlayerCount()+1)*(p.getIndex()+1))).setY((int)(HEIGHT/2));
            // Send their starting details
            p.sendInputLine(""+gameManager.getPlayerCount()+" "+p.getIndex()+" "+p.fullTank.getX()+" "+p.fullTank.getY());
        }
    }

    @Override
    public void gameTurn(int turn) {
        // Send each player all of the player's current states, starting with their own.
        // In the format "ID ENERGY X Y ROTATION
        for (Player player : gameManager.getActivePlayers()) {
            player.sendInputLine(""+player.getIndex()+" "+player.energy+" "+player.fullTank.getX()+" "+player.fullTank.getY()+" "+player.tank.getRotation()%(2*Math.PI));
            for(Player opp : gameManager.getPlayers()){
                if(!player.equals(opp)) {
                    player.sendInputLine(""+opp.getIndex()+" "+opp.energy+" "+opp.fullTank.getX()+" "+opp.fullTank.getY()+" "+opp.tank.getRotation()%(2*Math.PI));
                }
            }
            player.execute();
        }

        for (Player player : gameManager.getActivePlayers()) {
            try {
                // Get the player's actions
                List<String> outputs = Arrays.asList(player.getOutputs().get(0).split(" "));
                if(outputs.size()!=4){
                    throw new Exception("Not enough outputs");
                }

                // Clamp speed between 0 and MAXSPEED
                int speed = Integer.parseInt(outputs.get(0));
                if(speed > MAXSPEED)    speed = MAXSPEED;
                else if(speed < 0)      speed = 0;

                // Clamp heading change to between -MAXTURN and MAXTURN
                double heading = Double.parseDouble(outputs.get(1));
                if(heading < -MAXTURN)      heading = -MAXTURN;
                else if(heading > MAXTURN)  heading = MAXTURN;

                // Get the turret rotation desired, and whether or not to fire
                double turretHeading = Double.parseDouble(outputs.get(2));
                String fire = outputs.get(3);

                // Update player position
                player.tank.setRotation(player.tank.getRotation() + heading);
                player.turret.setRotation(turretHeading);
                player.fullTank.setX((int)(player.fullTank.getX()+speed*Math.sin(player.tank.getRotation())));
                player.fullTank.setY((int)(player.fullTank.getY()-speed*Math.cos(player.tank.getRotation())));

                // Bounce them back onto the playing field
                if(player.fullTank.getX()<0)            player.fullTank.setX(0);
                else if(player.fullTank.getX()>WIDTH)   player.fullTank.setX(WIDTH);
                if(player.fullTank.getY()<0)            player.fullTank.setY(0);
                else if(player.fullTank.getY()>HEIGHT)  player.fullTank.setY(HEIGHT);

                // If they have fired, add a bullet to the field
                if(fire.equals("FIRE")){
                    Sprite bullet = graphicEntityModule.createSprite().setImage("bullet.png")
                            .setAnchor(.5).setX(player.fullTank.getX()).setY(player.fullTank.getY()).setRotation(player.turret.getRotation());
                    bullets.add(bullet);
                    firedBy.add(player);
                    player.energy -= BULLETCOST;

                    // Handle a player firing away their last energy
                    if(player.energy<=0){
                        player.deactivate(player.getNicknameToken() + " Has exploded");
                        gameManager.addToGameSummary(player.getNicknameToken() + " Has exploded");
                    }
                }
                tooltips.setTooltipText(player.fullTank,"Energy: "+player.energy+"\nPosition (X,Y,Rot): ("+
                        player.fullTank.getX()+","+player.fullTank.getY()+","+player.fullTank.getRotation()+")");

            } catch (TimeoutException e) {
                // Took too long
                player.energy = 0;
                tooltips.setTooltipText(player.fullTank,"Energy: "+player.energy+"\nPosition (X,Y,Rot): ("+
                        player.fullTank.getX()+","+player.fullTank.getY()+","+player.fullTank.getRotation()+")");
                player.deactivate(player.getNicknameToken() + " Turn took too long.");
            }catch (NumberFormatException e){
                // Bad player action string
                player.energy = 0;
                tooltips.setTooltipText(player.fullTank,"Energy: "+player.energy+"\nPosition (X,Y,Rot): ("+
                        player.fullTank.getX()+","+player.fullTank.getY()+","+player.fullTank.getRotation()+")");
                player.deactivate(player.getNicknameToken() + " Invalid output types");
            }
            catch (Exception e) {
                // Something else bad happened
                player.energy = 0;
                tooltips.setTooltipText(player.fullTank,"Energy: "+player.energy+"\nPosition (X,Y,Rot): ("+
                        player.fullTank.getX()+","+player.fullTank.getY()+","+player.fullTank.getRotation()+")");
                player.deactivate(player.getNicknameToken() + " " + e.getClass());
            }
        }

        // Bullet collision
        for(int i=0;i<bullets.size();i++){
            Sprite bullet = bullets.get(i);
            Player firer = firedBy.get(i);

            // Move the bullet forward
            bullet.setX((int)(bullet.getX()+BULLETSPEED*Math.sin(bullet.getRotation())));
            bullet.setY((int)(bullet.getY()-BULLETSPEED*Math.cos(bullet.getRotation())));

            // Remove bullets that have gone past the playing area
            if(bullet.getX() < 0 || bullet.getX() > WIDTH || bullet.getY() < 0 || bullet.getY() > HEIGHT){
                bullets.get(i).setVisible(false);
                bullets.remove(i);
                firedBy.remove(i);
            }
            else {
                for (Player player : gameManager.getActivePlayers()) {
                    // If the bullet wasn't fired by this player, and it is within range, it hits
                    if (!player.equals(firer) && Math.hypot(Math.abs(bullet.getX() - player.fullTank.getX()), Math.abs(bullet.getY() - player.fullTank.getY())) < HITBOX) {
                        gameManager.addToGameSummary(player.getNicknameToken() + " was hit by " + firer.getNicknameToken());
                        player.energy -= BULLETDMG;
                        if (firer.isActive()) {
                            firer.energy += BULLETHEAL;
                            firer.score += BULLETDMG;
                        }
                        // If the bullet destroyed the player, remove them, and reward the firer
                        if (player.energy <= 0) {
                            tooltips.setTooltipText(player.fullTank, "Energy: " + player.energy + "\nPosition (X,Y,Rot): (" +
                                    player.fullTank.getX() + "," + player.fullTank.getY() + "," + player.fullTank.getRotation() + ")");
                            player.deactivate(player.getNicknameToken() + " Has exploded");
                            gameManager.addToGameSummary(player.getNicknameToken() + " Has exploded");
                            firer.score += (int) (MAXENERGY / 2);
                        }

                        // turn bullets into explosions for at least some cool effects
                        bullets.remove(i);
                        firedBy.remove(i);
                        bullet.setImage("explosion.png");
                        explosions.add(bullet);
                        explosionCounter.add(EXPLODEPERSIST);
                        i -= 1;
                        break;
                    }
                }
            }
        }

        // Explosions last for 3 turns
        for(int i=0;i<explosionCounter.size();i++){
            explosionCounter.set(i,explosionCounter.get(i)-1);
            if(explosionCounter.get(i)<=0){
                explosions.get(i).setVisible(false);
                explosions.remove(i);
                explosionCounter.remove(i);
                i-=1;
            }
        }

        if(gameManager.getActivePlayers().size()==1){
            gameManager.endGame();
        }
    }

    @Override
    public void onEnd() {
        for (Player p : gameManager.getPlayers()) {
            p.setScore(Math.max((int)(p.score+p.energy),0));
        }
        endScreenModule.setScores(gameManager.getPlayers().stream().mapToInt(AbstractMultiplayerPlayer::getScore).toArray());
    }
}
