
//import com.codingame.game.engine.Constants;


import java.util.Random;
import java.util.Scanner;


class Grid
{
  private boolean[][] walls;
  public int width;
  public int height;
  public int open = 0;

  public static Grid loadFromCGInput(Scanner in)
  {
    Grid grid = new Grid();
    grid.width = in.nextInt();
    grid.height = in.nextInt();
    grid.walls = new boolean[grid.width][grid.height];
    for (int y=0; y < grid.height; y++)
    {
      String row = in.next();
      for (int x=0; x < grid.width; x++)
      {
        grid.walls[x][y] = row.charAt(x)=='#';
        if (!grid.walls[x][y]) grid.open++;
      }
    }
    return grid;
  }

  public boolean isWall(int x, int y)
  {
    return x<0 || y <0 || x >= width || y >= height || walls[x][y];
  }

  @Override
  public String toString()
  {
    StringBuilder sb = new StringBuilder();
    sb.append(width).append(' ').append(height).append('\n');

    for (int y=0; y<height; y++)
    {
      for (int x=0; x<width; x++)
      {
        sb.append(walls[x][y] ? '#' : '.');
      }
      sb.append('\n');
    }

    return sb.toString();
  }
}

class JPS
{
  private Grid grid;
  private boolean[][][] jumpPoints;

  public Integer[][][] distance;

  public JPS(Grid grid)
  {
    this.grid = grid;
    jumpPoints = new boolean[grid.width][grid.height][8];
    distance = new Integer[grid.width][grid.height][8];
  }

  static class Directions
  {
    public static int N  = 0;
    public static int NE = 1;
    public static int E  = 2;
    public static int SE = 3;
    public static int S  = 4;
    public static int SW = 5;
    public static int W  = 6;
    public static int NW = 7;
  }
  
  public void preprocessing()
  {
    computePrimaryJumpPoints(true);
    computeStraightJumpPointsWestward();
    computeStraightJumpPointsEastward();
    computeStraightJumpPointsNorthward();
    computeStraightJumpPointsSouthward();
    computeDiagonalJumpPointsNorthwest();
    computeDiagonalJumpPointsNortheast();
    computeDiagonalJumpPointsSouthwest();
    computeDiagonalJumpPointsSoutheast();
  }

  private void computePrimaryJumpPoints(boolean verbose)
  {
    if (verbose) System.err.println("Primary Jump Points:");
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        // Traveling eastward
        if (!grid.isWall(x - 1, y) && (!grid.isWall(x, y - 1) && grid.isWall(x - 1, y - 1) || !grid.isWall(x, y + 1) && grid.isWall(x - 1, y + 1)))
        {
          jumpPoints[x][y][Directions.E] = true;
          if (verbose) System.err.println(String.format("  %d %d %s", x, y, "E"));
        }
        // Traveling westward
        if (!grid.isWall(x + 1, y) && (!grid.isWall(x, y - 1) && grid.isWall(x + 1, y - 1) || !grid.isWall(x, y + 1) && grid.isWall(x + 1, y + 1)))
        {
          jumpPoints[x][y][Directions.W] = true;
          if (verbose) System.err.println(String.format("  %d %d %s", x, y, "W"));
        }
        // Traveling northward
        if (!grid.isWall(x, y + 1) && (!grid.isWall(x - 1, y) && grid.isWall(x - 1, y + 1) || !grid.isWall(x + 1, y) && grid.isWall(x + 1, y + 1)))
        {
          jumpPoints[x][y][Directions.N] = true;
          if (verbose) System.err.println(String.format("  %d %d %s", x, y, "N"));
        }
        // Traveling southward
        if (!grid.isWall(x, y - 1) && (!grid.isWall(x - 1, y) && grid.isWall(x - 1, y - 1) || !grid.isWall(x + 1, y) && grid.isWall(x + 1, y - 1)))
        {
          jumpPoints[x][y][Directions.S] = true;
          if (verbose) System.err.println(String.format("  %d %d %s", x, y, "S"));
        }
      }
    }
  }

  private void computeStraightJumpPointsWestward()
  {
    for (int y = 0; y < grid.height; y++)
    {
      int count = -1;
      boolean jumpPointLastSeen = false;
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y))
        {
          count = -1;
          jumpPointLastSeen = false;
          distance[x][y][Directions.W] = 0;
          continue;
        }
        count++;
        if (jumpPointLastSeen)
        {
          distance[x][y][Directions.W] = count;
        } else // Wall last seen
        {
          distance[x][y][Directions.W] = -count;
        }
        if (jumpPoints[x][y][Directions.W])
        {
          count = 0;
          jumpPointLastSeen = true;
        }
      }
    }
  }

  private void computeStraightJumpPointsEastward()
  {
    for (int y = 0; y < grid.height; y++)
    {
      int count = -1;
      boolean jumpPointLastSeen = false;
      for (int x = grid.width - 1; x >= 0; x--)
      {
        if (grid.isWall(x, y))
        {
          count = -1;
          jumpPointLastSeen = false;
          distance[x][y][Directions.E] = 0;
          continue;
        }
        count++;
        if (jumpPointLastSeen)
        {
          distance[x][y][Directions.E] = count;
        } else // Wall last seen
        {
          distance[x][y][Directions.E] = -count;
        }
        if (jumpPoints[x][y][Directions.E])
        {
          count = 0;
          jumpPointLastSeen = true;
        }
      }
    }
  }

  private void computeStraightJumpPointsNorthward()
  {
    for (int x = 0; x < grid.width; x++)
    {
      int count = -1;
      boolean jumpPointLastSeen = false;
      for (int y = 0; y < grid.height; y++)
      {
        if (grid.isWall(x, y))
        {
          count = -1;
          jumpPointLastSeen = false;
          distance[x][y][Directions.N] = 0;
          continue;
        }
        count++;
        if (jumpPointLastSeen)
        {
          distance[x][y][Directions.N] = count;
        } else // Wall last seen
        {
          distance[x][y][Directions.N] = -count;
        }
        if (jumpPoints[x][y][Directions.N])
        {
          count = 0;
          jumpPointLastSeen = true;
        }
      }
    }
  }

  private void computeStraightJumpPointsSouthward()
  {
    for (int x = 0; x < grid.width; x++)
    {
      int count = -1;
      boolean jumpPointLastSeen = false;
      for (int y = grid.height - 1; y >= 0; y--)
      {
        if (grid.isWall(x, y))
        {
          count = -1;
          jumpPointLastSeen = false;
          distance[x][y][Directions.S] = 0;
          continue;
        }
        count++;
        if (jumpPointLastSeen)
        {
          distance[x][y][Directions.S] = count;
        } else // Wall last seen
        {
          distance[x][y][Directions.S] = -count;
        }
        if (jumpPoints[x][y][Directions.S])
        {
          count = 0;
          jumpPointLastSeen = true;
        }
      }
    }
  }

  private void computeDiagonalJumpPointsNorthwest()
  {
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        if (grid.isWall(x - 1, y) || grid.isWall(x, y - 1) || grid.isWall(x - 1, y - 1))
        {
          distance[x][y][Directions.NW] = 0; // Wall one away
        } else if (!grid.isWall(x - 1, y) && !grid.isWall(x, y - 1) && (distance[x - 1][y - 1][Directions.N] > 0 || distance[x - 1][y - 1][Directions.W] > 0))
        {
          distance[x][y][Directions.NW] = 1; // Straight jump point one away
        } else
        {
          int jumpDistance = distance[x - 1][y - 1][Directions.NW]; // Increment from last
          distance[x][y][Directions.NW] = jumpDistance + (jumpDistance > 0 ? 1 : -1);
        }
      }
    }
  }

  private void computeDiagonalJumpPointsNortheast()
  {
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        if (grid.isWall(x + 1, y) || grid.isWall(x, y - 1) || grid.isWall(x + 1, y - 1))
        {
          distance[x][y][Directions.NE] = 0; // Wall one away
        } else if (!grid.isWall(x + 1, y) && !grid.isWall(x, y - 1) && (distance[x + 1][y - 1][Directions.N] > 0 || distance[x + 1][y - 1][Directions.E] > 0))
        {
          distance[x][y][Directions.NE] = 1; // Straight jump point one away
        } else
        {
          int jumpDistance = distance[x + 1][y - 1][Directions.NE]; // Increment from last
          distance[x][y][Directions.NE] = jumpDistance + (jumpDistance > 0 ? 1 : -1);
        }
      }
    }
  }

  private void computeDiagonalJumpPointsSouthwest()
  {
    for (int y = grid.height - 1; y >= 0; y--)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        if (grid.isWall(x - 1, y) || grid.isWall(x, y + 1) || grid.isWall(x - 1, y + 1))
        {
          distance[x][y][Directions.SW] = 0; // Wall one away
        } else if (!grid.isWall(x - 1, y) && !grid.isWall(x, y + 1) && (distance[x - 1][y + 1][Directions.S] > 0 || distance[x - 1][y + 1][Directions.W] > 0))
        {
          distance[x][y][Directions.SW] = 1; // Straight jump point one away
        } else
        {
          int jumpDistance = distance[x - 1][y + 1][Directions.SW]; // Increment from last
          distance[x][y][Directions.SW] = jumpDistance + (jumpDistance > 0 ? 1 : -1);
        }
      }
    }
  }

  private void computeDiagonalJumpPointsSoutheast()
  {
    for (int y = grid.height - 1; y >= 0; y--)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        if (grid.isWall(x + 1, y) || grid.isWall(x, y + 1) || grid.isWall(x + 1, y + 1))
        {
          distance[x][y][Directions.SE] = 0; // Wall one away
        } else if (!grid.isWall(x + 1, y) && !grid.isWall(x, y + 1) && (distance[x + 1][y + 1][Directions.S] > 0 || distance[x + 1][y + 1][Directions.E] > 0))
        {
          distance[x][y][Directions.SE] = 1; // Straight jump point one away
        } else
        {
          int jumpDistance = distance[x + 1][y + 1][Directions.SE]; // Increment from last
          distance[x][y][Directions.SE] = jumpDistance + (jumpDistance > 0 ? 1 : -1);
        }
      }
    }
  }

  public String toPreprocessingCGInput()
  {
    StringBuilder sb = new StringBuilder();
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;
        sb.append(x).append(' ').append(y);
        sb.append(' ').append(distance[x][y][Directions.N]);
        sb.append(' ').append(distance[x][y][Directions.NE]);
        sb.append(' ').append(distance[x][y][Directions.E]);
        sb.append(' ').append(distance[x][y][Directions.SE]);
        sb.append(' ').append(distance[x][y][Directions.S]);
        sb.append(' ').append(distance[x][y][Directions.SW]);
        sb.append(' ').append(distance[x][y][Directions.W]);
        sb.append(' ').append(distance[x][y][Directions.NW]);
        sb.append('\n');
      }
    }
    return sb.toString();
  }
}

public class Player
{
  public static void main(String[] args) throws InterruptedException
  {
    Grid grid = Grid.loadFromCGInput(new Scanner(System.in));
    System.err.println(grid);

    JPS jps = new JPS(grid);
    jps.preprocessing();

    System.out.println(jps.toPreprocessingCGInput());
  }
}
