package com.codingame.game.engine;


import com.codingame.game.Player;
import com.codingame.gameengine.core.SoloGameManager;
import com.codingame.gameengine.module.entities.*;
import com.codingame.gameengine.module.tooltip.TooltipModule;

import java.util.ArrayList;
import java.util.HashMap;

import static com.codingame.game.engine.Constants.*;

public class Viewer
{

  private GraphicEntityModule graphic;
  private SoloGameManager<Player> manager;
  private TooltipModule tooltip;

  private Grid grid;

  private int TILE_MARGIN, MIN_X, MIN_Y, TILESIZE;
  private HashMap<Integer, Integer> DIR_CENTER_X = new HashMap<>();
  private HashMap<Integer, Integer> DIR_CENTER_Y = new HashMap<>();

  private Text[][][] distances;
  private ArrayList<Rectangle> mistakes;

  public Viewer(GraphicEntityModule graphic, SoloGameManager<Player> manager, TooltipModule tooltip)
  {
    this.graphic = graphic;
    this.manager = manager;
    this.tooltip = tooltip;
  }


  public void init(Grid grid)
  {
    this.grid = grid;

    TILESIZE = Math.min((VIS_SCREEN_WIDTH - 2 * VIS_GLOBAL_MARGIN) / grid.width,
            (VIS_SCREEN_HEIGHT - 2 * VIS_GLOBAL_MARGIN) / grid.height);
    MIN_X = (VIS_SCREEN_WIDTH - (TILESIZE * grid.width)) / 2;
    MIN_Y = (VIS_SCREEN_HEIGHT - (TILESIZE * grid.height)) / 2;
    TILE_MARGIN = Math.min(VIS_TILE_MARGIN_MAX, (int) Math.max(1, TILESIZE * VIS_TILE_MARGIN_FRAC));

    DIR_CENTER_X.put(Directions.N,  MIN_X + TILESIZE/2);
    DIR_CENTER_Y.put(Directions.N,  MIN_Y + TILESIZE/6);
    DIR_CENTER_X.put(Directions.NE, MIN_X + TILESIZE - TILESIZE/6);
    DIR_CENTER_Y.put(Directions.NE, MIN_Y + TILESIZE/6);
    DIR_CENTER_X.put(Directions.E,  MIN_X + TILESIZE - TILESIZE/6);
    DIR_CENTER_Y.put(Directions.E,  MIN_Y + TILESIZE/2);
    DIR_CENTER_X.put(Directions.SE, MIN_X + TILESIZE - TILESIZE/6);
    DIR_CENTER_Y.put(Directions.SE, MIN_Y + TILESIZE - TILESIZE/6);
    DIR_CENTER_X.put(Directions.S,  MIN_X + TILESIZE/2);
    DIR_CENTER_Y.put(Directions.S,  MIN_Y + TILESIZE - TILESIZE/6);
    DIR_CENTER_X.put(Directions.SW,  MIN_X + TILESIZE/6);
    DIR_CENTER_Y.put(Directions.SW,  MIN_Y + TILESIZE - TILESIZE/6);
    DIR_CENTER_X.put(Directions.W,   MIN_X + TILESIZE/6);
    DIR_CENTER_Y.put(Directions.W,   MIN_Y + TILESIZE/2);
    DIR_CENTER_X.put(Directions.NW,  MIN_X + TILESIZE/6);
    DIR_CENTER_Y.put(Directions.NW,  MIN_Y + TILESIZE/6);

    distances = new Text[grid.width][grid.height][8];
    mistakes = new ArrayList<>();

    //System.out.println(TILE_MARGIN);
    //System.out.println(TILESIZE);
  }

  public void showGrid()
  {
    graphic.createRectangle().setX(0).setWidth(VIS_SCREEN_WIDTH)
            .setY(0).setHeight(VIS_SCREEN_HEIGHT)
            .setFillColor(0x000000).setZIndex(-2);

    graphic.createRectangle().setX(MIN_X - TILE_MARGIN).setWidth(TILESIZE * grid.width + 2 * TILE_MARGIN)
            .setY(MIN_Y - TILE_MARGIN).setHeight(TILESIZE * grid.height + 2 * TILE_MARGIN)
            .setFillColor(COL_PASSABLE_TILE).setZIndex(-1);

    for (int x = 0; x < grid.width + 1; x++)
    {
      graphic.createLine().setX(MIN_X + x * TILESIZE).setY(MIN_Y)
              .setX2(MIN_X + x * TILESIZE).setY2(MIN_Y + TILESIZE * grid.height)
              .setLineWidth(VIS_BORDER_WIDTH).setLineColor(COL_BORDERS).setZIndex(0);
    }

    for (int y = 0; y < grid.height + 1; y++)
    {
      graphic.createLine().setX(MIN_X).setY(MIN_Y + y * TILESIZE)
              .setX2(MIN_X + TILESIZE * grid.width).setY2(MIN_Y + y * TILESIZE)
              .setLineWidth(VIS_BORDER_WIDTH).setLineColor(COL_BORDERS).setZIndex(0);
    }

    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        Rectangle tile = graphic.createRectangle()
                .setX(MIN_X + TILESIZE * x + TILE_MARGIN)
                .setWidth(TILESIZE - 2 * TILE_MARGIN)
                .setY(MIN_Y + TILESIZE * y + TILE_MARGIN)
                .setHeight(TILESIZE - 2 * TILE_MARGIN)
                .setZIndex(0);
        if (grid.isWall(x, y))
        {
          tile.setFillColor(COL_WALL_TILE);
          tooltip.setTooltipText(tile, String.format("column = %d\nrow = %d\nWALL", x, y));
        }
        else
        {
          tile.setFillColor(COL_PASSABLE_TILE);
          tooltip.setTooltipText(tile, String.format("column = %d\nrow = %d", x, y));
        }
      }
    }
  }

  public void prepareDistances(Engine engine, int d)
  {
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        String val = engine.playerAnswer[x][y][d];
        Text t = graphic.createText(val)
                .setX(DIR_CENTER_X.get(d) + x * TILESIZE)
                .setY(DIR_CENTER_Y.get(d) + y * TILESIZE)
                .setFontSize(TILESIZE/4)
                .setAnchor(0.5)
                .setFillColor(0x000000)
                .setZIndex(1)
                .setAlpha(0);
        String msg = String.format("direction = %s\ngiven distance = %s", ReverseDirections.get(d), val);
        if (engine.playerMistakes.isEmpty())
          msg += "\n──────────\nCORRECT";
        tooltip.setTooltipText(t, msg);
        distances[x][y][d] = t;
      }
    }
    graphic.commitWorldState(0);
  }

  public void showDistances()
  {
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        for (int d = 0; d < 8; d++)
        {
          if (distances[x][y][d] != null)
          {
            distances[x][y][d].setAlpha(1);
          }
        }
      }
    }
    graphic.commitWorldState(0);
  }

  public void prepareMistakes(Engine engine, int d)
  {
    int half = TILESIZE/8;
    for (int y = 0; y < grid.height; y++)
    {
      for (int x = 0; x < grid.width; x++)
      {
        if (grid.isWall(x, y)) continue;

        Boolean correct = engine.isAnswerCorrect(x, y, d);
        Rectangle hint = graphic.createRectangle()
                .setX(DIR_CENTER_X.get(d) + x * TILESIZE - half)
                .setWidth(2*half)
                .setY(DIR_CENTER_Y.get(d) + y * TILESIZE - half)
                .setHeight(2*half)
                .setFillColor(correct?COL_CORRECT:COL_WRONG)
                .setZIndex(0)
                .setAlpha(0);;
        if (correct)
          tooltip.setTooltipText(hint, "CORRECT");
        else
          tooltip.setTooltipText(hint, String.format("MISTAKE\ncorrect distance = %d", engine.jps.distance[x][y][d]));
        mistakes.add(hint);
      }
    }
    graphic.commitWorldState(0);
  }

  public void showMistakes()
  {
    for (Rectangle hint:mistakes)
    {
      hint.setAlpha(1);
    }
    graphic.commitWorldState(0);
  }

  public void showSummary(boolean passed)
  {

    graphic.commitWorldState(0);

    Sprite end = graphic.createSprite().setImage(passed?"passed.png":"failed.png").setX(960).setY(570)
            .setScale(40.8).setZIndex(3).setAlpha(1).setAnchor(0.5);
    graphic.commitEntityState(0, end);
    end.setScale(0.5);
    graphic.commitEntityState(0.1, end);
    //win.setScale(1);
    //score.setScale(0.9);
    //graphic.commitEntityState(0.4, win, score);
    //win.setScale(1.1);
    //score.setScale(1.0);
    //graphic.commitEntityState(0.5, win, score);


  }
}
