package com.codingame.game;

import com.codingame.game.engine.*;
import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.SoloGameManager;
import com.codingame.gameengine.module.tooltip.TooltipModule;
import com.codingame.gameengine.module.entities.*;
import com.google.inject.Inject;

import java.util.List;

public class Referee extends AbstractReferee
{
  @Inject private SoloGameManager<Player> manager;
  @Inject private GraphicEntityModule graphic;
  @Inject private TooltipModule tooltip;

  private Engine engine;
  private Viewer viewer;
  private Player player;

  @Override
  public void init()
  {
    engine = new Engine(manager);

    player = manager.getPlayer();
    player.expectedOutputLines = engine.grid.open;

    manager.setFirstTurnMaxTime(Constants.TIMELIMIT_INIT);
    //manager.setTurnMaxTime(Constants.TIMELIMIT_TURN);
    manager.setMaxTurns(Constants.ACTIONLIMIT);
    manager.setFrameDuration(Constants.FRAMEDURATIONS[0]);

    viewer = new Viewer(graphic, manager, tooltip);
    viewer.init(engine.grid);
    viewer.showGrid();
  }

  @Override
  public void gameTurn(int turn)
  {
    manager.setFrameDuration(Constants.FRAMEDURATIONS[turn]);

    if (turn==1)
    {
      manager.addTooltip(manager.getPlayer(), "Map");
      if (!executePlayer())
      {
        return;
      }
    }

    if (turn >= 1 && turn <= 8) // Preload distance entities
    {
      viewer.prepareDistances(engine, turn - 1);
    }

    if (turn == 8) // Show distance entities
    {
      viewer.showDistances();
      manager.addTooltip(manager.getPlayer(), "Given distances");
    }

    if (turn==9 && engine.playerMistakes.isEmpty()) // Summary visualization if correct
    {
      manager.setFrameDuration(Constants.FRAMEDURATION_SUMMARY);
      viewer.showSummary(true);
      manager.winGame("PASSED!");
      return;
    }

    if (turn > 8 && turn <= 16) // Preload mistake entities
    {
      viewer.prepareMistakes(engine, turn - 9);
    }

    if (turn == 16) // Show mistake entities
    {
      viewer.showMistakes();
      manager.addTooltip(manager.getPlayer(), "Mistakes");
    }

    if (turn == 17) // Summary visualization if mistakes
    {
      manager.setFrameDuration(Constants.FRAMEDURATION_SUMMARY);
      viewer.showSummary(false);
      manager.loseGame(engine.mistakesMessage());
    }
  }

  @Override
  public void onEnd()
  {
    //manager.putMetadata("Score", ""+gameState.score);
  }

  private boolean executePlayer()
  {
    try
    {
      String[] input = engine.grid.toString().trim().split("\n");
      for (int i=0; i < input.length; i++)
        manager.getPlayer().sendInputLine(input[i]);

      manager.getPlayer().execute();

      List<String> outputs = manager.getPlayer().getOutputs();
      engine.validateAnswer(outputs);
    }
    catch (ActionErrorException e)
    {
      manager.setFrameDuration(Constants.FRAMEDURATION_SUMMARY);
      viewer.showSummary(false);
      manager.loseGame("[ERROR] "+e.getMessage());
      return false;
    }
    catch (TimeoutException e)
    {
      manager.setFrameDuration(Constants.FRAMEDURATION_SUMMARY);
      viewer.showSummary(false);
      manager.loseGame("[ERROR] Timeout!");
      return false;
    }
    catch (Exception e)
    {
      manager.setFrameDuration(Constants.FRAMEDURATION_SUMMARY);
      viewer.showSummary(false);
      manager.loseGame(String.format("[ERROR] Output not properly formatted. Should be %d lines: 'column row N NE E SE S SW W NW'.", player.getExpectedOutputLines()));
      return false;
    }
    return true;
  }
}
