package com.codingame.game;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.endscreen.EndScreenModule;
import com.codingame.gameengine.module.entities.Sprite;
import com.codingame.gameengine.module.entities.Text;
import com.codingame.gameengine.module.entities.TextBasedEntity.TextAlign;
import com.codingame.gameengine.module.tooltip.TooltipModule;
import com.google.inject.Inject;

public class Referee extends AbstractReferee {
	// Uncomment the line below and comment the line under it to create a Solo
	// Game
	// @Inject private SoloGameManager<Player> gameManager;
	@Inject
	private MultiplayerGameManager<Player> gameManager;
	@Inject
	private GraphicEntityModule graphicEntityModule;
	@Inject
	private EndScreenModule endScreenModule;
	@Inject
	private TooltipModule tooltipModule;

	public enum _LEAGUE {
		WOOD3, WOOD2, WOOD1, BRONZE
	}

	Game gameBoard;
	Viewer gameView;
	Player currentPlayer;
	int nbPlayers = 0;
	int league;

	private static final int FIRST_TURN_MAX_TIME = 1000; // ms
	private static final int OTHER_TURN_MAX_TIME = 50; // ms

	private static final int MAX_MESSAGE_LENGTH = 20;

	static final Pattern PLAYER_MOVE_PATTERN = Pattern.compile(
			"^(?<action>MOVE)\\s+(?<id>-?\\d+)\\s+(?<iceblock>[A-Z]{1}[0-9]{1})(?:\\s+MSG((\\s*)(?<message>.+)))?",
			Pattern.CASE_INSENSITIVE);
	
	static final Pattern PLAYER_MOVE_PUSH_PATTERN = Pattern.compile(
			"^(MOVE)\\s+(?<id>[0-3]{1})\\s+(?<iceblock>[A-H]{1}[1-8]{1})(\\s+(?<push>PUSH))?(?:\\s+MSG((\\s*)(?<message>.*)))?",
			Pattern.CASE_INSENSITIVE);

	static String EXPECTED = "MOVE <id> <xy> or MOVE <id> <xy> PUSH <xy>";

	/********/
	/* init */
	/********/

	@Override
	public void init() {

		try {

			// Players
			nbPlayers = gameManager.getPlayerCount();

			// Get league
			league = gameManager.getLeagueLevel();

			gameView = new Viewer(graphicEntityModule, gameManager, tooltipModule);
			
			long seed = gameManager.getSeed();
			gameBoard = new Game(nbPlayers, seed, league, gameView);

			// Initialize your game here.
			gameManager.setMaxTurns(60 + nbPlayers * gameBoard.numberOfPenguins[nbPlayers]);
			gameManager.setTurnMaxTime(FIRST_TURN_MAX_TIME);

			// Initialize viewer
			initializeView();

			currentPlayer = gameManager.getPlayer(0);

			// initial input: number of players, player id and number of penguins for each player
			// input grid size
			for (Player p : gameManager.getPlayers()) {
				p.sendInputLine(String.valueOf(gameManager.getPlayerCount()));
				p.sendInputLine(String.valueOf(p.getIndex()));
				p.sendInputLine(String.valueOf(gameBoard.numberOfPenguins[nbPlayers]));
			}
		} catch (Exception e) {
			e.printStackTrace();
			System.err.println("Referee failed to initialize");
			abort();
		}
	}

	private void abort() {
		System.err.println("Unexpected game end");
		gameManager.endGame();
	}

	/************/
	/* gameTurn */
	/************/

	@Override
	public void gameTurn(int turn) {

		String outputLine = "";
		
		if (turn <= nbPlayers) {
			gameManager.setTurnMaxTime(FIRST_TURN_MAX_TIME);
		} else {
			gameManager.setTurnMaxTime(OTHER_TURN_MAX_TIME);
		}

		try {

			List<String> actions = gameBoard.listPlayablesActions(currentPlayer.getIndex());

			if (actions.size() == 0) {
				return;
			}

			sendPlayerInputs(currentPlayer, currentPlayer.getIndex(), turn, actions);
			currentPlayer.execute();

			List<String> outputs = currentPlayer.getOutputs();

			// Check validity of the player output and compute the new game
			// state
			outputLine = outputs.get(0);
			
			Pattern pattern = (league == 1) ? PLAYER_MOVE_PATTERN : PLAYER_MOVE_PUSH_PATTERN ;
			
			Matcher matchMove = pattern.matcher(outputLine);
			if (matchMove.matches()) {
				//System.err.println("match move OK");

				String idPenguin = matchMove.group("id");
				String iceblock = matchMove.group("iceblock");
				String push = (league == 1) ? null : matchMove.group("push");
				String message = matchMove.group("message");

				if (message == null) {
					message = "";
				}
				else if (message.length() > MAX_MESSAGE_LENGTH) {
					message = message.substring(0, MAX_MESSAGE_LENGTH);
				}

				//gameManager.addToGameSummary(String.format("%s played %s %s %s", currentPlayer.getNicknameToken(), action, idPenguin, iceblock));

				//System.err.println("call playAction");
				Game.Action actionResult = gameBoard.playAction(currentPlayer, idPenguin, iceblock, push, actions);
				if (actionResult == null) {
					throw new Game.InvalidPlay();
				}
				//System.err.println("playAction OK");

				if (actionResult.source == null) {
					gameManager.addToGameSummary(String.format("%s played Move penguin#%d to %s%s",
						currentPlayer.getNicknameToken(),
						actionResult.penguinId,
						actionResult.dest,
						(actionResult.pushSource == null) ? "" : (" and push penguin " + actionResult.pushSource + " ")));
				} else {
					gameManager.addToGameSummary(String.format("%s played Move penguin#%d %s to %s%sScore : +%d",
						currentPlayer.getNicknameToken(),
						actionResult.penguinId,
						actionResult.source,
						actionResult.dest,
						(actionResult.pushSource == null) ? " " : (" and push penguin " + actionResult.pushSource + " "),
						actionResult.score));
				}
				
				if (actionResult.pushSource  != null) {
					gameManager.addTooltip(currentPlayer, String.format("%s push Penguin in %s !", currentPlayer.getNicknameToken(), actionResult.pushSource));
				}
						
				currentPlayer.messageHub.setText(message);
				currentPlayer.scoreHub.setText(String.valueOf(currentPlayer.getScore()));

				gameView.drawMove(gameBoard, actionResult);

			} else {
				//System.err.println("match move NOK");
				throw new Game.InvalidInput();
			}
		} catch (TimeoutException e) {
			onError(currentPlayer, "timeout", "did not respond in time", "Timeout");
		} catch (NumberFormatException e) {
			onError(currentPlayer, "invalid input: " + outputLine, "gave invalid input: " + outputLine, "Invalid input");
		} catch (Game.InvalidInput invalidInput) {
			onError(currentPlayer, "invalid input: " + outputLine, "gave invalid input: " + outputLine, "Invalid input");
		} catch (Game.InvalidPlay invalidPlay) {
			onError(currentPlayer, "invalid move: " + outputLine, "played invalid move: " + outputLine, "Invalid move");
		} catch (Exception e) {
			System.out.println(outputLine);
			onError(currentPlayer, "Error", "Error", "Error");
		}

		// next player

		currentPlayer = getNextPlayer(currentPlayer.getIndex());

		if (currentPlayer == null) {

			// If no player can move, it is the end of the game

			ArrayList<Integer> IdWinners = getIdWinners();

			for (int id : IdWinners) {
				Player winner = gameManager.getPlayer(id);
				String exaequo = (IdWinners.size() == 1) ? "" : "ex aequo";
				gameManager.addTooltip(winner, winner.getNicknameToken() + " wins " + exaequo + "!");
				gameManager.addToGameSummary(String.format("%s won the game!", winner.getNicknameToken()));
				winner.scoreHub.setText(winner.getScore() + " - WIN");
			}

			gameManager.endGame();
		}
	}

	/***********/
	/* onError */
	/***********/

	void onError(Player player, String textError1, String textError2, String textError3) {
		player.deactivate(String.format("$%d %s !", player.getIndex(), textError1));
		gameManager.addToGameSummary(String.format("%s %s !", player.getNicknameToken(), textError2));
		player.scoreHub.setText(textError3 + " !").setFillColor(0xff0000);
		//player.setScore(player.getScore() % 100);
		player.setScore(-1);
	}

	/*****************/
	/* getNextPlayer */
	/*****************/

	// If no player or end of the game return null
	
	public Player getNextPlayer(int currentIndex) {

		int nextIdPlayer = ((currentIndex + 1) % nbPlayers);
		for (int i = 0; i < nbPlayers; i++) {
			int idPlayer = (nextIdPlayer + i) % nbPlayers;
			Player player = gameManager.getPlayer(idPlayer);

			if (player.isActive() == false) continue;

			List<String> actions = gameBoard.listPlayablesActions(idPlayer);
			
			if (actions.size() != 0) {
				
				if (currentIndex == idPlayer) {
					// If only one player, wait until the player is winning
					int scorePlayer = gameManager.getPlayer(idPlayer).getScore();
					for (int j = 0; j < nbPlayers; j++) {
						if (j == idPlayer) continue;
						if (gameManager.getPlayer(j).getScore() >= scorePlayer) {
							// we find a blocked player with a best or equal score
							return (gameManager.getPlayer(idPlayer));
						}
					}
					
					// End of the game
					return(null);
				}
				
				return (gameManager.getPlayer(idPlayer));
			}
		}

		return (null);
	}

	/****************/
	/* getIdWinners */
	/****************/

	public ArrayList<Integer> getIdWinners() {
		ArrayList<Integer> IdWinners = new ArrayList<Integer>();
		int ScoreMax = Integer.MIN_VALUE;
		for (int i = 0; i < nbPlayers; i++) {
			if (gameManager.getPlayer(i).getScore() > ScoreMax) {
				ScoreMax = gameManager.getPlayer(i).getScore();
				IdWinners.clear();
			}

			if (gameManager.getPlayer(i).getScore() == ScoreMax) {
				IdWinners.add(i);
			}
		}

		return (IdWinners);
	}
	
	/********************/
	/* sendPlayerInputs */
	/********************/

	void sendPlayerInputs(Player player, int who, int turn, List<String> actions) {

		StringBuilder sb = new StringBuilder();

		// Ice block map

		player.sendInputLine(String.valueOf(Game.HEIGHT));
		for (int y = 0; y < Game.HEIGHT; y++) {
			sb = new StringBuilder();
			int width = (y % 2 == 0) ? Game.WIDTH0 : Game.WIDTH1;
			for (int x = 0; x < width; x++) {
				IceBlock iceBlock = gameBoard.iceBlockGrid[y][x];
				sb.append(iceBlock.getFishQuantity());
			}

			player.sendInputLine(sb.toString());
		}

		// Score
		// player.sendInputLine(String.valueOf(gameManager.getPlayerCount()));
		for (int i = 0; i < gameManager.getPlayerCount(); i++) {
			player.sendInputLine(String.valueOf(gameManager.getPlayer(i).getScore()));
		}

		// Penguins
		player.sendInputLine(String.valueOf(gameBoard.penguins.size()));
		for (Penguin penguin : gameBoard.penguins) {
			sb = new StringBuilder();
			
			sb.append(String.valueOf(penguin.id) + " ");
			sb.append(String.valueOf(penguin.playerId) + " ");
			
			if (penguin.isPushed() == false) {			
				String sCoord = String.format("%s", penguin.iceBlock.pos);
				sb.append(sCoord);			
			} else {
				sb.append("??");
			}

			player.sendInputLine(sb.toString());
		}

		// Send the actions.

		player.sendInputLine(String.valueOf(actions.size()));
		for (String s : actions) {
			player.sendInputLine(s);
		}
	}

	/******************/
	/* initializeView */
	/******************/

	private void initializeView() {
		gameView.init();

		for (int y = 0; y < Game.HEIGHT; y++) {
			int width = (y % 2 == 0) ? Game.WIDTH0 : Game.WIDTH1;
			for (int x = 0; x < width; x++) {
				IceBlock iceBlock = gameBoard.iceBlockGrid[y][x];
				gameView.drawFish(x, y, iceBlock.getFishQuantity());
			}
		}

		drawHud();
	}

	/*********/
	/* onEnd */
	/*********/

	@Override
	public void onEnd() {
		// show final ranking
		endScreenModule.setTitleRankingsSprite("logo.png");
		int[] scores = new int[nbPlayers];
		for (int i = 0; i < nbPlayers; i++) {
			scores[i] = gameManager.getPlayer(i).getScore();
		}
		endScreenModule.setScores(scores);
	}

	//////////////////////////////////////////////

	private void drawHud() {

		for (Player player : gameManager.getPlayers()) {
			// 0 .. 1920
			int x = (player.getIndex() % 2 == 0) ? 190 : 435 + 1050 + 40 + 190;
			int y = (player.getIndex() < 2) ? 150 : 620;

			graphicEntityModule.createRectangle().setWidth(140).setHeight(140).setX(x - 70).setY(y - 70).setLineWidth(0)
					.setFillColor(player.getColorToken());
			graphicEntityModule.createRectangle().setWidth(120).setHeight(120).setX(x - 60).setY(y - 60).setLineWidth(0)
					.setFillColor(0xffffff);

			Text text = graphicEntityModule.createText(player.getNicknameToken()).setX(x).setY(y + 120).setZIndex(20)
					.setFontSize(40).setFillColor(0xffffff).setAnchor(0.5);

			Sprite avatar = graphicEntityModule.createSprite().setX(x).setY(y).setZIndex(20)
					.setImage(player.getAvatarToken()).setAnchor(0.5).setBaseHeight(116).setBaseWidth(116);

			Text messageEntity = 
					graphicEntityModule.createText("")
					.setX(x).setY(y + 200)
					//.setZIndex(20)
					.setFontSize(36)
					.setMaxWidth(375)
					.setFillColor(0xffffff)
					.setAnchor(0.5);
						
			Text winnerText = graphicEntityModule.createText("").setX(x).setY(y + 280).setZIndex(20).setFontSize(40)
					.setFillColor(player.getColorToken()).setFontWeight(Text.FontWeight.BOLD).setAnchor(0.5);

			player.messageHub = messageEntity;
			player.scoreHub = winnerText;
		}
	}
}
