package com.codingame.game;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.Random;

/********/
/* Game */
/********/

public class Game {

	private int nbPlayers = 2;
	private int league = 0;
	public int[] numberOfPenguins = { 0, 0, 4, 3, 2 };

	public static final int WIDTH0 = 7;
	public static final int WIDTH1 = 8;
	public static final int HEIGHT = 8;

	private ArrayList<IceBlock> iceBlocks = new ArrayList<IceBlock>();
	public IceBlock iceBlockGrid[][] = new IceBlock[HEIGHT][WIDTH1];

	public ArrayList<Penguin> penguins = new ArrayList<Penguin>();
	private Map<String, Penguin> mapPenguins = new HashMap<String, Penguin>();
	
	// Direction between 2 ice blocks
	private int lineDirection[][][][] = new int[HEIGHT][WIDTH1][HEIGHT][WIDTH1];
	
	Viewer viewer;

	public static class InvalidInput extends Exception {
		public InvalidInput() {
		}
	}

	public static class InvalidPlay extends Exception {
		public InvalidPlay() {
		}
	}

	/**********/
	/* Action */
	/**********/

	public class Action {

		int playerId;
		int penguinId;
		
		int dir;
		
		Coordinate source;
		Coordinate dest;
		
		Coordinate pushSource;
		Coordinate pushDest;
		Penguin pushedPenguin;
		
		int score = 0;

		public Action(int playerId, int penguinId, int dir, Coordinate source, Coordinate dest, Coordinate pushSource, Coordinate pushDest, Penguin pushedPenguin) {
			this.playerId = playerId;
			this.penguinId = penguinId;
			this.dir = dir;
			this.source = source;
			this.dest = dest;
			this.pushSource = pushSource;
			this.pushDest = pushDest;
			this.pushedPenguin = pushedPenguin;
			this.score = 0;
		}
	}

	/********/
	/* Game */
	/********/

	public Game(int nbPlayers, long seed, int league, Viewer viewer) {
		this.nbPlayers = nbPlayers;
		this.league = league;
		this.viewer = viewer;
		
		init(seed);
	}

	/********/
	/* init */
	/********/

	public void init(long seed) {

		// Create the ice blocks
		for (int y = 0; y < HEIGHT; y++) {
			int width = (y % 2 == 0) ? WIDTH0 : WIDTH1;
			for (int x = 0; x < width; x++) {

				IceBlock iceBlock = new IceBlock(x, y);
				iceBlocks.add(iceBlock);
				iceBlockGrid[y][x] = iceBlock;
			}
		}

		// Quantity of fishes

		ArrayList<Integer> fishes = new ArrayList<Integer>();
		for (int i = 0; i < 60; i++) {
			if (i < 10)
				fishes.add(3);
			else if (i < 30)
				fishes.add(2);
			else
				fishes.add(1);
		}

		// shuffle fishes

		final Random generator = new Random(seed);
		Collections.shuffle(fishes, generator);

		int i = 0;
		for (IceBlock iceBlock : iceBlocks) {
			iceBlock.fishQuantity = fishes.get(i);
			i++;
		}

		// Neighbors of each ice block

		Coordinate[] DeltaCoordinate = { new Coordinate(+1, -1, 0), new Coordinate(0, -1, +1),
				new Coordinate(-1, 0, +1), new Coordinate(-1, +1, 0), new Coordinate(0, +1, -1),
				new Coordinate(+1, 0, -1) };

		// Compute neighbours
		
		// lineDirection : direction between 2 ice block is set to -1 by default
		for (int x0 = 0; x0 < WIDTH1; x0++) {
			for (int y0 = 0; y0 < HEIGHT; y0++) {
				for (int x1 = 0; x1 < WIDTH1; x1++) {
					for (int y1 = 0; y1 < HEIGHT; y1++) {
						lineDirection[y0][x0][y1][x1] = -1;
					}
				}
			}
		}
		
		for (IceBlock iceBlock : iceBlocks) {

			// System.err.print(iceBlock.pos + " : ");

			for (int dir = 0; dir < 6; dir++) {
				Coordinate Delta = DeltaCoordinate[dir];
				Coordinate dest = new Coordinate(iceBlock.getPos().getX() + Delta.getX(),
												 iceBlock.getPos().getY() + Delta.getY(),
												 iceBlock.getPos().getZ() + Delta.getZ());

				if (dest.col < 0 || dest.row < 0 || dest.col >= WIDTH1 || dest.row >= HEIGHT)
					continue;

				IceBlock iceBlockDest = iceBlockGrid[dest.row][dest.col];
				if (iceBlockDest != null) {
					// System.err.print(iceBlockDest.pos + " ");
					iceBlock.addNeighbours(dir, iceBlockDest);
				}
			}

			// lineDirection : direction between 2 ice block
			for (int x0 = 0; x0 < WIDTH1; x0++) {
				for (int y0 = 0; y0 < HEIGHT; y0++) {
					IceBlock iceBlockSource = iceBlockGrid[y0][x0];
					if (iceBlockSource == null) continue;
					for (int iDir = 0; iDir < 6; iDir++) {
						IceBlock iceBlockDest = iceBlockSource.neighbours[iDir];
						while (iceBlockDest != null) {
							lineDirection[y0][x0][iceBlockDest.pos.row][iceBlockDest.pos.col] = iDir;
							iceBlockDest = iceBlockDest.neighbours[iDir];
						}
					}
				}
			}
			
			// System.err.println();
		}
	}

	/************************/
	/* listPlayablesActions */
	/************************/

	List<String> listPlayablesActions(int playerId) {

		ArrayList<String> actions = new ArrayList<String>();

		// Create myPenguins extract from penguins
		ArrayList<Penguin> myPenguins = new ArrayList<Penguin>();
		for (Penguin penguin : penguins) {
			if (penguin.playerId == playerId) {
				myPenguins.add(penguin);
			}
		}

		// all my penguins is in the game ?
		if (myPenguins.size() < numberOfPenguins[nbPlayers]) {

			// all my penguins is not in the game
			int idPenguin = myPenguins.size();

			for (IceBlock iceBlock : iceBlocks) {
				if (iceBlock.getFishQuantity() != 0 && iceBlock.penguin == null) {
					String sAction = String.format("MOVE %d %s", idPenguin, iceBlock.pos);
					actions.add(sAction);								
				}
			}
		} else {
			// all my penguins is in the game
			for (Penguin myPenguin : myPenguins) {

				if (myPenguin.isPushed() == true) continue;

				IceBlock iceBlockSource = myPenguin.getIceBlock();

				IceBlock iceBlockDest = null;
				for (int iDir = 0; iDir < 6; iDir++) {
					iceBlockDest = iceBlockSource.neighbours[iDir];
					while (iceBlockDest != null && iceBlockDest.isAvailable()) {
						String sAction = String.format("MOVE %d %s", myPenguin.id, iceBlockDest.pos);
						actions.add(sAction);
						
						if (league >= 2) {				
							Penguin penguinInEdge = iceBlockDest.getNearPenguinInEdge(playerId, myPenguin.id, iDir);
							if (penguinInEdge != null) {
								//actions.add(sAction + " PUSH " + penguinInEdge.iceBlock.pos);
								actions.add(sAction + " PUSH");
							}
						}

						iceBlockDest = iceBlockDest.neighbours[iDir];
					}
				}
			}
		}

		return (actions);
	}

	/**************/
	/* playAction */
	/**************/

	public Action playAction(Player player, String sPenguinId, String iceblock, String sPush, List<String> actions) {

		int playerId = player.getIndex();
		int penguinId = Integer.parseInt(sPenguinId);
		if (penguinId < 0 || penguinId >= numberOfPenguins[nbPlayers]) {
			// Error
			return (null);
		}

		// Construct string actions
		
		String sAction = "MOVE " + sPenguinId + " " + iceblock;
		if (sPush != null) {
			sAction += " PUSH";
		}
		
		// Legal action ?
		
		// action is legal if it is present in the list of playable actions.
		
		//System.err.println("Legal action ?");		
		boolean bActionFound = false;
		for (String sLegalAction : actions) {
			if (sAction.equals(sLegalAction)) {
				bActionFound = true;
				break;
			}
		}
		
		// if action not found in the list of playable action : error
		if (bActionFound == false) {
			//System.err.println("not legal");
			return(null);
		}
		
		// build the action with command string
		// action contains data for the simulation and the viewer
		
		Penguin penguin = null;
		
		// Ice Block source 
		
		//System.err.println("Ice Block source");
		
		IceBlock iceBlockSource = null;
		Coordinate posSource = null;
		String key = String.format("%d_%d", playerId, penguinId);
		
		if (mapPenguins.containsKey(key) == true) {
			penguin = mapPenguins.get(key);
			iceBlockSource = penguin.iceBlock;
			posSource = iceBlockSource.pos;
		}
		
		// Ice Block Dest
		
		//System.err.println("Ice Block dest");
		
		int col = (int) (iceblock.charAt(0) - 'A');
		int row = (int) (iceblock.charAt(1) - '1');

		Coordinate posDest = new Coordinate(col, row);
		IceBlock iceBlockDest = iceBlockGrid[row][col];
		
		int dir = getPlayedDirection(iceBlockSource, iceBlockDest);

		// Push : Coordinate of the pushed penguin
		
		//System.err.printf("Coordinate of the pushed penguin dir:%d [%s]\n", dir, iceBlockDest.toString());
		
		Coordinate posPush = null;
		if (sPush != null) {
			if (iceBlockDest.neighbours[dir] != null) {
				posPush = iceBlockDest.neighbours[dir].getPos();
			}
		}
		
		// Is it a push ?
		
		//System.err.println("push possible ?");
		
		Coordinate pushDest = null;
		Penguin pushedPenguin = null;
		if (posPush != null) {
			IceBlock iceBlockPush = iceBlockGrid[posPush.row][posPush.col];
			iceBlockPush.penguin.iceBlock = iceBlockPush.getSeaDestination(iceBlockDest);
			
			// null : it is the sea not null : it is a hole (iceblock removed)
			if (iceBlockPush.penguin.iceBlock != null) {
				// information for the viewer
				Coordinate posPinguinPushed = iceBlockPush.penguin.iceBlock.pos;
				pushDest = new Coordinate(posPinguinPushed.col, posPinguinPushed.row);
			}
			
			// penguin is pushed
			iceBlockPush.penguin.setPenguinPushed();
			pushedPenguin = iceBlockPush.penguin;
			iceBlockPush.penguin = null;
		}
		
		// Create an action structure with the data
		
		//System.err.println("Create an action structure with the data");
				
		Action action = new Action(playerId, penguinId, dir, posSource, posDest, posPush, pushDest, pushedPenguin);
		
		// Play the move

		if (iceBlockSource == null) {

			// It is a new penguin
			
			//System.err.println("It is a new penguin");
			
			penguin = new Penguin(penguinId, playerId, iceBlockDest);
			iceBlockDest.setPenguin(penguin);

			mapPenguins.put(key, penguin);
			penguins.add(penguin);

		} else {

			// The penguin exists, play the movement and update score

			//System.err.println("Action : " + key + " " + action.source + " -> " + action.dest);

			int score = 100 * penguin.iceBlock.getFishQuantity() + 1;

			player.updateScore(score);
			action.score = score;

			penguin.iceBlock.setFishQuantity(0);
			penguin.iceBlock.setPenguin(null);

			penguin.iceBlock = iceBlockDest;
			iceBlockDest.setPenguin(penguin);
		}
		
		//System.err.println("end palyAction");

		return (action);
	}
	
	/**********************/
	/* getPlayedDirection */
	/**********************/

	private int getPlayedDirection(IceBlock iceblockSource, IceBlock iceBlockDest) {
		if (iceblockSource == null || iceBlockDest == null) {
			return(-1);
		}
		
		int dir = lineDirection[iceblockSource.pos.row][iceblockSource.pos.col][iceBlockDest.pos.row][iceBlockDest.pos.col];
				
		return(dir);
	}
}

/**************/
/* Coordinate */
/**************/

class Coordinate {
	int col, row;
	int x, y, z;

	/**************/
	/* Coordinate */
	/**************/

	public Coordinate(int x, int y, int z) {
		this.x = x;
		this.y = y;
		this.z = z;

		Convert2D();
	}

	Coordinate(int col, int row) {
		this.col = col;
		this.row = row;

		this.x = col - (row + (row & 1)) / 2;
		this.z = row;
		this.y = -(this.x + row);
	}

	public int getX() {
		return (x);
	}

	public int getY() {
		return (y);
	}

	public int getZ() {
		return (z);
	}

	@Override
	public String toString() {
		return (String.format("%c%c", 'A' + col, '1' + row));
	}

	/*************/
	/* Convert2D */
	/*************/

	void Convert2D() {
		col = x + (z + (z & 1)) / 2;
		row = z;
	}
	
	/*******/
	/* add */
	/*******/
	
	public static Coordinate add(Coordinate A, Coordinate B) {
		return(new Coordinate(A.x + B.x, A.y + B.y, A.z + B.z));
	}
	
	/*******/
	/* sub */
	/*******/
	
	public static Coordinate sub(Coordinate A, Coordinate B) {
		return(new Coordinate(A.x - B.x, A.y - B.y, A.z - B.z));
	}
};

/***********/
/* Penguin */
/***********/

class Penguin {
	int id;
	int playerId;
	IceBlock iceBlock = null;
	boolean pushed = false;

	public Penguin(int id, int playerId, IceBlock iceBlock) {
		this.id = id;
		this.playerId = playerId;
		this.iceBlock = iceBlock;
		this.pushed = false;
	}

	@Override
	public String toString() {
		return ("Penguin [id=" + id + ", playerId=" + playerId + ", pushed=" + pushed + "]");
	}

	public IceBlock getIceBlock() {
		return (iceBlock);
	}

	public boolean isPushed() {
		return (pushed == true);
	}
	
	public void setPenguinPushed() {
		this.pushed = true;
	}

	public void setIceBlock(IceBlock iceBlock) {
		this.iceBlock = iceBlock;
	}

	public String getKey() {
		return (String.format("%d_%d", playerId, id));
	}
}

/************/
/* IceBlock */
/************/

class IceBlock {

	Coordinate pos;
	int fishQuantity = 0;
	IceBlock[] neighbours = { null, null, null, null, null, null };
	Penguin penguin = null;

	public IceBlock(int x, int y) {
		this.pos = new Coordinate(x, y);
	}

	@Override
	public String toString() {
		return "IceBlock [pos=" + pos + ", fishQuantity=" + fishQuantity + "]";
	}

	public Penguin getPenguin() {
		return penguin;
	}

	public void setPenguin(Penguin penguin) {
		this.penguin = penguin;
	}

	public int getFishQuantity() {
		return (fishQuantity);
	}

	public void setFishQuantity(int fishQuantity) {
		this.fishQuantity = fishQuantity;
	}

	public Coordinate getPos() {
		return pos;
	}

	public void setPos(Coordinate pos) {
		this.pos = pos;
	}

	public void addNeighbours(int dir, IceBlock iceBlock) {
		this.neighbours[dir] = iceBlock;
	}
	
	/**************/
	/* isIsolated */
	/**************/

	public boolean isIsolated() {
		for (int dir = 0; dir < 6; dir++) {
			if (neighbours[dir] != null)
				return (false);
		}
		return (true);
	}
	
	/**********/
	/* isEdge */
	/**********/
	
	public boolean isEdge() {
		for (int dir = 0; dir < 6; dir++) {
			if (neighbours[dir] == null) return(true);
			if (neighbours[dir].getFishQuantity() == 0) return (true);
		}
		return (false);
	}
	
	public boolean isEdge(int dir) {
		if (neighbours[dir] == null) return(true);
		if (neighbours[dir].getFishQuantity() == 0) return (true);
		return (false);
	}
	
	/*********************/
	/* getSeaDestination */
	/*********************/
	
	// getSeaDestination : null if it is the sea or the iceblock if it is a future sea hole
	
	public IceBlock getSeaDestination(IceBlock iceBlockDest) {
		for (int dir = 0; dir < 6; dir++) {
			if (neighbours[dir] == iceBlockDest) {
				// return hex on opposite direction
				return(neighbours[(dir + 3) % 6]);
			}
		}

		return (null);
	}
	
	/************************/
	/* getNearPenguinInEdge */
	/************************/
	
	public ArrayList<Penguin> getNearPenguinInEdge(int playerId, int penguinId) {
		ArrayList<Penguin> penguinsInEdge = new ArrayList<Penguin>();
			
		for (int dir = 0; dir < 6; dir++) {
			if (neighbours[dir] != null &&
				neighbours[dir].penguin != null && neighbours[dir].penguin.isPushed() == false &&
				// not the same penguin
				(neighbours[dir].penguin.playerId != playerId || neighbours[dir].penguin.id != penguinId) &&
				neighbours[dir].getFishQuantity() > 0 &&
				neighbours[dir].isEdge()) {
				penguinsInEdge.add(neighbours[dir].penguin);
			}
		}
		
		return(penguinsInEdge);
	}
	
	public Penguin getNearPenguinInEdge(int playerId, int penguinId, int dir) {

		if (neighbours[dir] != null &&
			neighbours[dir].penguin != null && neighbours[dir].penguin.isPushed() == false &&
			// not the same penguin
			(neighbours[dir].penguin.playerId != playerId || neighbours[dir].penguin.id != penguinId) &&
			neighbours[dir].getFishQuantity() > 0 &&
			neighbours[dir].isEdge(dir)) {
			return(neighbours[dir].penguin);
		}
		
		return(null);
	}
	
	/***************/
	/* isAvailable */
	/***************/

	public boolean isAvailable() {
		return (fishQuantity != 0 && penguin == null);
	}
}