package com.codingame.game;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.entities.Sprite;
import com.codingame.gameengine.module.entities.Text;
import com.google.inject.Inject;

public class Referee extends AbstractReferee {
    // Uncomment the line below and comment the line under it to create a Solo Game
    // @Inject private SoloGameManager<Player> gameManager;
    @Inject private MultiplayerGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;

    private static final int BOARDSIZE_LEVEL1 = 2;
    private static final int BOARDSIZE_LEVEL2 = 7;
    private int boardSize;

    Game gameBoard;
    Viewer gameView;
    int currPlayerIndex = 0;

    private static final int TURN_MAX_TIME = 100; // ms

    private static final Pattern INPUT_MOVE_PATTERN = Pattern.compile("(?<box>[A-Z]{1}[0-9]{1})(\\s*)(?<side>[LTRB]{1})(?:\\s+MSG((\\s*)(?<message>.+)))?");
    private static final int MAX_MESSAGE_LENGTH = 20;

    @Override
    public void init() {
        // Initialize your game here.
        gameManager.setTurnMaxTime(TURN_MAX_TIME);
        if (gameManager.getLeagueLevel() == 1) {
            boardSize = BOARDSIZE_LEVEL1;
        } else {
            boardSize = BOARDSIZE_LEVEL2;
        }
        gameManager.setMaxTurns(2 * boardSize * (boardSize + 1));
        gameView = new Viewer(graphicEntityModule, boardSize,
                gameManager.getPlayer(0).getColorToken(),
                gameManager.getPlayer(1).getColorToken());
        gameBoard = new Game(boardSize, gameView);
        gameView.init();
        drawHud();
    }

    @Override
    public void gameTurn(int turn) {
        Player player = gameManager.getPlayer(currPlayerIndex);
        Player opponent = gameManager.getPlayer(1 - currPlayerIndex);
        sendPlayerInputs(player, player.getIndex(), turn);
        player.execute();
        try {
            List<String> outputs = player.getOutputs();
            // Check validity of the player output and compute the new game state
            String outputLine = outputs.get(0);
            Matcher matchMove = INPUT_MOVE_PATTERN.matcher(outputLine);
            if (matchMove.matches()) {
                String box = matchMove.group("box");
                String side =  matchMove.group("side");
                String message = matchMove.group("message");
                if (message == null)
                    message = "";
                else if (message.length() > MAX_MESSAGE_LENGTH) {
                    message = message.substring(0, MAX_MESSAGE_LENGTH);
                }
                Game.PlayResult result = gameBoard.play(currPlayerIndex, box, side);
                gameManager.addToGameSummary(String.format("%s played %s %s", player.getNicknameToken(), box, side));
                if (!result.takenBoxes.isEmpty()) {
                    String msg = String.format("%s takes", player.getNicknameToken());
                    for (String s : result.takenBoxes) {
                        msg += " " + s;
                    }
                    gameManager.addToGameSummary(msg);
                    gameManager.addTooltip(player, msg);
                }
                player.messageHub.setText(message);
                player.scoreHub.setText(String.valueOf(gameBoard.getScore(player.getIndex())));
                opponent.scoreHub.setText(String.valueOf(gameBoard.getScore(opponent.getIndex())));
                currPlayerIndex = result.nextPlayerIndex;
            } else {
                throw new Game.InvalidInput();
            }
            if (gameBoard.isFinal()) {
                Player player0 = gameManager.getPlayer(0);
                Player player1 = gameManager.getPlayer(1);
                player0.setScore(gameBoard.getScore(0));
                player1.setScore(gameBoard.getScore(1));
                if (gameBoard.isAWin(0)) {
                    Player winner = gameManager.getPlayer(0);
                    Player looser = gameManager.getPlayer(1);
                    gameManager.addTooltip(player, winner.getNicknameToken() + " wins!");
                    gameManager.addToGameSummary(String.format("%s won the game!", winner.getNicknameToken()));
                    String winText = String.format("%d - WIN", gameBoard.getScore(0));
                    winner.scoreHub.setText(winText);
                } else if (gameBoard.isAWin(1)) {
                    Player winner = gameManager.getPlayer(1);
                    Player looser = gameManager.getPlayer(0);
                    gameManager.addTooltip(player, winner.getNicknameToken() + " wins!");
                    gameManager.addToGameSummary(String.format("%s won the game!", winner.getNicknameToken()));
                    String winText = String.format("%d - WIN", gameBoard.getScore(1));
                    winner.scoreHub.setText(winText);
                } else {
                    gameManager.addToGameSummary("The game is a draw!");
                    String drawText = String.format("%d - DRAW", gameBoard.getScore(0));
                    player0.scoreHub.setText(drawText);
                    player1.scoreHub.setText(drawText);
                }
                gameManager.endGame();
            }
        } catch (TimeoutException e) {
            player.deactivate(String.format("$%d timeout!", player.getIndex()));
            gameManager.addToGameSummary(String.format("%s did not respond in time!", player.getNicknameToken()));
            player.setScore(-1);
            player.scoreHub.setText("Timeout!").setFillColor(0xff0000);
            opponent.scoreHub.setText("WIN");
            gameManager.endGame();
        } catch (NumberFormatException e) {
            player.deactivate(String.format("$%d invalid input!", player.getIndex()));
            gameManager.addToGameSummary(String.format("%s gave invalid input!", player.getNicknameToken()));
            player.setScore(-1);
            player.scoreHub.setText("Invalid input!").setFillColor(0xff0000);
            opponent.scoreHub.setText("WIN");
            gameManager.endGame();
        } catch (Game.InvalidInput invalidInput) {
            player.deactivate(String.format("$%d invalid input!", player.getIndex()));
            gameManager.addToGameSummary(String.format("%s gave invalid input!", player.getNicknameToken()));
            player.setScore(-1);
            player.scoreHub.setText("Invalid input!").setFillColor(0xff0000);
            opponent.scoreHub.setText("WIN");
            gameManager.endGame();
        } catch (Game.InvalidPlay invalidPlay) {
            player.deactivate(String.format("$%d invalid move!", player.getIndex()));
            gameManager.addToGameSummary(String.format("%s played invalid move!", player.getNicknameToken()));
            player.setScore(-1);
            player.scoreHub.setText("Invalid move!").setFillColor(0xff0000);
            opponent.scoreHub.setText("WIN");
            gameManager.endGame();
        }
    }

    void sendPlayerInputs(Player player, int who, int turn) {
        // Send board size and player ID at first turn.
        if (turn <= 2) {
            player.sendInputLine(String.valueOf(boardSize));
            player.sendInputLine(String.valueOf((char)('A' + who)));
        }
        // Send the player and the opponent's score.
        int playerIndex = player.getIndex();
        player.sendInputLine(String.format("%d %d", gameBoard.getScore(playerIndex), gameBoard.getScore(1 - playerIndex)));
        // Send the number of playable boxes, adn their names and sides.
        List<String> playables = gameBoard.listPlayablesBoxes();
        player.sendInputLine(String.valueOf(playables.size()));
        for (String s : playables) {
            player.sendInputLine(s);
        }
    }

    private void drawHud() {
        for (Player player : gameManager.getPlayers()) {
            int x = player.getIndex() == 0 ? 220 : 1920 - 220;
            int y = 220;
            graphicEntityModule
                    .createRectangle()
                    .setWidth(140)
                    .setHeight(140)
                    .setX(x - 70)
                    .setY(y - 70)
                    .setLineWidth(0)
                    .setFillColor(player.getColorToken());
            graphicEntityModule
                    .createRectangle()
                    .setWidth(120)
                    .setHeight(120)
                    .setX(x - 60)
                    .setY(y - 60)
                    .setLineWidth(0)
                    .setFillColor(0xffffff);
            Text text = graphicEntityModule.createText(player.getNicknameToken())
                    .setX(x)
                    .setY(y + 120)
                    .setZIndex(20)
                    .setFontSize(40)
                    .setFillColor(0xffffff)
                    .setAnchor(0.5);
            Sprite avatar = graphicEntityModule.createSprite()
                    .setX(x)
                    .setY(y)
                    .setZIndex(20)
                    .setImage(player.getAvatarToken())
                    .setAnchor(0.5)
                    .setBaseHeight(116)
                    .setBaseWidth(116);
            Text messageEntity = graphicEntityModule.createText("")
                    .setX(x)
                    .setY(y + 200)
                    .setZIndex(20)
                    .setFontSize(40)
                    .setFillColor(0xffffff)
                    .setAnchor(0.5);
            Text winnerText = graphicEntityModule.createText("")
                    .setX(x)
                    .setY(y + 280)
                    .setZIndex(20)
                    .setFontSize(60)
                    .setFillColor(0xffff00)
                    .setFontWeight(Text.FontWeight.BOLD)
                    .setAnchor(0.5);
            player.messageHub = messageEntity;
            player.scoreHub = winnerText;
        }
    }
}
