package com.codingame.game;
import java.util.*;


public class Game {

    private int boardSize;
    private int[] scores = {0, 0};
    ArrayList<Box> boxes;
    Viewer viewer;

    public Game(int boardSize, Viewer viewer) {
        this.boardSize = boardSize;
        this.viewer = viewer;
        // Create the array of boxes.
        Box.setBoardSize(boardSize);
        boxes = new ArrayList<Box>(boardSize * boardSize);
        for (int c = 0; c < boardSize; c++) {
            for (int r = 0; r < boardSize; r++) {
                Box box = new Box(c, r);
                boxes.add(box);
            }
        }
    }

    public int getScore(int who) {
        return scores[who];
    }

    private Box findBoxName(String name) {
        Box box = null;
        for (Box b : boxes) {
            if (b.getName().equals(name)) {
                box = b;
                break;
            }
        }
        return box;
    }

    public class PlayResult {
        int nextPlayerIndex;
        List<String> takenBoxes;
    }

    public PlayResult play(int who, String boxName, String side) throws InvalidInput, InvalidPlay {
        PlayResult result = new PlayResult();
        result.takenBoxes = new ArrayList<String>();
        // Find the box from its name.
        Box box = findBoxName(boxName);
        if (box == null) {
            throw new InvalidPlay();
        }
        // Verify that the side can be played.
        if (!box.isPlayableSide(side)) {
            throw new InvalidPlay();
        }
        // Remove the playable side from the box as well as from its neighbour.
        // Increment the score of the player if he closes a box.
        boolean samePlayer = false;
        box.removePlayableSide(side);
        viewer.drawMove(who, boxName + " " + side);
        if (!box.isPlayable()) {
            samePlayer = true;
            scores[who] += 1;
            viewer.drawOwnedBox(who, box.getName());
            result.takenBoxes.add(box.getName());
        }
        Box.BoxSide bs = box.findNeighbor(side);
        if (bs != null) {
            Box box2 = findBoxName(bs.boxName);
            box2.removePlayableSide(bs.side);
            if (!box2.isPlayable()) {
                samePlayer = true;
                scores[who] += 1;
                viewer.drawOwnedBox(who, box2.getName());
                result.takenBoxes.add(box2.getName());
            }
        }
        // Return the index of the player for next play. Same player plays
        // again when a box is closed.
        result.nextPlayerIndex = samePlayer ? who : (1 - who);
        return result;
    }

    public boolean isFinal() {
        return ((scores[0] + scores[1]) == (boardSize * boardSize));
    }

    public boolean isAWin(int who) {
        return (scores[who] > scores[1 - who]);
    }

    public static class InvalidInput extends Exception {
        public InvalidInput() {
        }
    }

    public static class InvalidPlay extends Exception {
        public InvalidPlay() {
        }
    }

    List<String> listPlayablesBoxes() {
        ArrayList<String> lines = new ArrayList<String>();
        for (Box box : boxes) {
            if (!box.isPlayable()) {
                continue;
            }
            String line = String.format("%s %s", box.getName(), box.getPlayableSides());
            lines.add(line);
        }
        return lines;
    }
}

class Box {
    private String name;
    private String playableSides;
    private static int boardSize;

    public Box(int col, int row) {
        this.name = String.format("%c%d", (char)('A' + col), (row + 1));
        this.playableSides = "LTRB";
    }

    public static void setBoardSize(int n) { boardSize = n; }

    public String getName() { return name; }

    public String getPlayableSides() { return playableSides; }

    public boolean isPlayable() {
        return !playableSides.isEmpty();
    }

    public boolean isPlayableSide(String side) {
        return playableSides.contains(side);
    }
    public void removePlayableSide(String side) {
        playableSides = playableSides.replace(side, "");
    }

    public class BoxSide {
        String boxName;
        String side;
    }

    public BoxSide findNeighbor(String side) {
        int col = name.charAt(0) - 'A';
        int row = name.charAt(1) - '1';
        if (side.equals("L")) {
            if (col == 0) {
                return null;
            }
            BoxSide bs = new BoxSide();
            bs.boxName = String.format("%c%d", (char)('A' + col - 1), (row + 1));
            bs.side = "R";
            return bs;
        } else if (side.equals("R")) {
            if (col == (boardSize - 1)) {
                return null;
            }
            BoxSide bs = new BoxSide();
            bs.boxName = String.format("%c%d", (char)('A' + col + 1), (row + 1));
            bs.side = "L";
            return bs;
        } else if (side.equals("B")) {
            if (row == 0) {
                return null;
            }
            BoxSide bs = new BoxSide();
            bs.boxName = String.format("%c%d", (char)('A' + col), (row + 1 - 1));
            bs.side = "T";
            return bs;
        } else if (side.equals("T")) {
            if (row == (boardSize - 1)) {
                return null;
            }
            BoxSide bs = new BoxSide();
            bs.boxName = String.format("%c%d", (char)('A' + col), (row + 1 + 1));
            bs.side = "B";
            return bs;
        }
        return null;
    }
}