@file:JvmName("Referee")

package com.codingame.game

import com.codingame.game.grids.GridBuilder
import com.codingame.game.grids.PlainGridBuilder
import com.codingame.game.grids.SmallQuartersGridBuilder
import com.codingame.gameengine.core.AbstractPlayer
import com.codingame.gameengine.core.AbstractReferee
import com.codingame.gameengine.core.MultiplayerGameManager
import com.codingame.gameengine.module.entities.GraphicEntityModule
import com.codingame.gameengine.module.tooltip.TooltipModule
import com.google.inject.Inject
import lombok.extern.java.Log

@Suppress("unused")
@Log
class Referee : AbstractReferee() {

    private val CANVAS_WIDTH = 1920
    private val CANVAS_HEIGHT = 1080

    private val LOGO_OFFSET = 20

    private val GRID_PADDING = 15

    private val GRID_MARGIN_LEFT = 450
    private val GRID_MARGIN_RIGHT = 50
    private val GRID_WIDTH = CANVAS_WIDTH - GRID_MARGIN_LEFT - GRID_MARGIN_RIGHT - 2 * GRID_PADDING

    private val GRID_MARGIN_TOP = 50
    private val GRID_MARGIN_BOTTOM = 50
    private val GRID_HEIGHT = CANVAS_HEIGHT - GRID_MARGIN_TOP - GRID_MARGIN_BOTTOM - 2 * GRID_PADDING

    private val USERS_MARGIN_LEFT = 15
    private val USER_MARGIN_BOTTOM = 15
    private val USER_AVATAR_BORDER = 3
    private val USER_AVATAR_SIZE = 114
    private val USER_AVATAR_CONTAINER_SIZE = USER_AVATAR_SIZE + 2 * USER_AVATAR_BORDER
    private val USER_TEXT_MARGIN_HORIZONTAL = 25
    private val USER_TEXT_MAX_WIDTH = 170
    private var USER_DOG_TAG_PADDING = 20
    private var USER_DOG_TAG_EXTRA_PADDING_LEFT = 30
    private val USER_CONTAINER_WIDTH = USER_AVATAR_SIZE + USER_TEXT_MAX_WIDTH + USER_DOG_TAG_EXTRA_PADDING_LEFT +
            2 * (USER_TEXT_MARGIN_HORIZONTAL + USER_DOG_TAG_PADDING)
    private val USER_CONTAINER_HEIGHT = USER_AVATAR_SIZE + 2 * USER_DOG_TAG_PADDING
    private val USER_NAME_FONT_SIZE = 27
    private val USER_NAME_HEIGHT = USER_NAME_FONT_SIZE * 4 / 3
    private val USER_SCORE_FONT_SIZE = 39
    private val USER_SCORE_HEIGHT = USER_SCORE_FONT_SIZE * 4 / 3
    private val USER_STATS_FONT_SIZE = 27
    private val USER_STATS_HEIGHT = USER_STATS_FONT_SIZE * 4 / 3

    @Inject
    private lateinit var gameManager: MultiplayerGameManager<Player>

    @Inject
    private lateinit var graphicEntityModule: GraphicEntityModule

    @Inject
    private lateinit var tooltipModule: TooltipModule

    @Inject
    private lateinit var gameProperties: GameProperties

    @Inject
    private lateinit var plainGridBuilder: PlainGridBuilder

    @Inject
    private lateinit var chessGridBuilder: SmallQuartersGridBuilder

    private lateinit var board: Board2D<Field>

    override fun init() {
        gameProperties.init()

        initBackground()

        val gridColumns = gameProperties.width
        val gridRows = gameProperties.height

        val gridMaxCellWidth = GRID_WIDTH / gridColumns
        val gridMaxCellHeight = GRID_HEIGHT / gridRows
        val gridCellSize = minOf(gridMaxCellWidth, gridMaxCellHeight)

        val gridOffsetX = GRID_MARGIN_LEFT + GRID_PADDING + (GRID_WIDTH - gridCellSize * gridColumns) / 2
        val gridOffsetY: Int = GRID_MARGIN_TOP + GRID_PADDING + (GRID_HEIGHT - gridCellSize * gridRows) / 2

        val gridBackground = graphicEntityModule.createRectangle()
        gridBackground.width = gridCellSize * gridColumns + 2 * GRID_PADDING
        gridBackground.height = gridCellSize * gridRows + 2 * GRID_PADDING
        gridBackground.x = gridOffsetX - GRID_PADDING
        gridBackground.y = gridOffsetY - GRID_PADDING
        gridBackground.fillColor = 0x777777
        gridBackground.fillAlpha = 0.50

        val gridTemplate = listOf(
            plainGridBuilder,
            chessGridBuilder
        )
            .filter(GridBuilder::isApplicable)
            .random(gameProperties.random)
            .createGrid()

        board = Board2D(gridColumns, gridRows) { x, y ->
            val sprite = graphicEntityModule.createSprite()
            sprite.image = "empty.png"
            sprite.anchorX = 0.5
            sprite.anchorY = 0.5
            sprite.x = (gridOffsetX + (x + 0.5) * gridCellSize).toInt()
            sprite.y = (gridOffsetY + (y + 0.5) * gridCellSize).toInt()
            sprite.baseWidth = (0.9 * gridCellSize).toInt()
            sprite.baseHeight = (0.9 * gridCellSize).toInt()
            sprite.tint = 0x000000
            gridTemplate[x, y].copy(sprite = sprite)
        }

        gameManager.players.forEach {
            val playerBackground = graphicEntityModule.createSprite()
            playerBackground.image = "dogtag.png"
            playerBackground.baseWidth = USER_CONTAINER_WIDTH
            playerBackground.baseHeight = USER_CONTAINER_HEIGHT
            val avatarSprite = graphicEntityModule.createSprite()
            avatarSprite.image = it.avatarToken
            avatarSprite.baseWidth = USER_AVATAR_SIZE
            avatarSprite.baseHeight = USER_AVATAR_SIZE
            avatarSprite.anchorX = 0.5
            avatarSprite.anchorY = 0.5
            avatarSprite.x = USER_CONTAINER_HEIGHT / 2 + USER_DOG_TAG_EXTRA_PADDING_LEFT
            avatarSprite.y = USER_CONTAINER_HEIGHT / 2
            val userName = graphicEntityModule.createText(it.nicknameToken)
            userName.fillColor = it.colorToken
            userName.x = avatarSprite.x + avatarSprite.baseWidth / 2 + USER_TEXT_MARGIN_HORIZONTAL
            userName.y = USER_DOG_TAG_PADDING
            userName.fontSize = USER_NAME_FONT_SIZE
            userName.maxWidth = USER_TEXT_MAX_WIDTH
            val userScore = graphicEntityModule.createText("0")
            userScore.fillColor = it.colorToken
            userScore.x = userName.x
            userScore.y = userName.y + USER_NAME_HEIGHT
            userScore.fontSize = USER_SCORE_FONT_SIZE
            userScore.maxWidth = USER_TEXT_MAX_WIDTH
            val userStats = graphicEntityModule.createText("0 / 0")
            userStats.fillColor = it.colorToken
            userStats.x = userScore.x
            userStats.y = userScore.y + USER_SCORE_HEIGHT
            userStats.fontSize = USER_STATS_FONT_SIZE
            userStats.maxWidth = USER_TEXT_MAX_WIDTH
            val userGroup = graphicEntityModule.createGroup(
                playerBackground,
                avatarSprite,
                userName,
                userScore,
                userStats
            )
            userGroup.x = USERS_MARGIN_LEFT

            it.scoreText = userScore
            it.statsText = userStats
            it.group = userGroup
        }

        initialInput()
        redraw()
    }

    override fun gameTurn(turn: Int) {
        if (turn % 2 == 1) {
            val villages = gameProperties.villages
            for (player in gameManager.activePlayers) {
                player.sendInputLine(villages.toString())
                boardInput(player)
                player.execute()
            }
            for (player in gameManager.activePlayers) {
                try {
                    player.outputs
                        .flatMap { line ->
                            line.split(' ')
                                .filter(String::isNotEmpty)
                                .map {
                                    try {
                                        it.toInt()
                                    } catch (e: NumberFormatException) {
                                        throw Player.InvalidOutputException("$it is not a number")
                                    }
                                }
                                .let { lineNumbers ->
                                    if (lineNumbers.size % 2 == 1) {
                                        throw Player.InvalidOutputException("Must output an even number of numbers. Instead ${lineNumbers.size} numbers")
                                    }
                                    if (villages * 2 >= lineNumbers.size) {
                                        (0 until lineNumbers.size / 2).map {
                                            Pair(lineNumbers[it * 2], lineNumbers[it * 2 + 1])
                                        }
                                    } else {
                                        throw Player.InvalidOutputException("Output did not contain at most ${2 * villages} numbers. Instead ${lineNumbers.size} numbers")
                                    }
                                }
                        }
                        .forEach {
                            val field = board[it.first, it.second]
                            if (null != field.owner && player != field.owner) {
                                throw Player.InvalidOutputException("Tried to place a village on foreign Field")
                            }
                            if (
                                (field.village is PlayerVillage && player != (field.village as PlayerVillage).owner)
                                ||
                                (field.village is NeutralVillage && (field.village as NeutralVillage).creation == turn)
                            ) {
                                field.village = NeutralVillage(turn) // Simultaneously
                            } else if (null != field.village) {
                                throw Player.InvalidOutputException("Tried to place a village on another")
                            } else {
                                field.village = PlayerVillage(player, turn)
                            }
                        }
                } catch (e: AbstractPlayer.TimeoutException) {
                    player.deactivate("${player.nicknameToken}: Timeout!")
                } catch (e: Player.InvalidOutputException) {
                    player.deactivate("${player.nicknameToken}: ${e.reason}")
                } catch (e: IndexOutOfBoundsException) {
                    player.deactivate("${player.nicknameToken}: Tried to place a village off the grid")
                }
            }
        } else {
            applyRules(turn)
        }

        capture()
        redraw()
    }

    private fun initialInput() {
        listOf(
            gameManager.playerCount.toString(),
            "${board.width} ${board.height}"
        )
            .forEach {
                for (player in gameManager.activePlayers) {
                    player.sendInputLine(it)
                }
            }
    }

    private fun boardInput(player: Player) {
        board.entries.map { line ->
            player.sendInputLine(line.joinToString(" ") {
                when (it.village) {
                    is NeutralVillage -> "N"
                    is PlayerVillage -> ((gameManager.playerCount + (it.village as PlayerVillage).owner.index - player.index) % gameManager.playerCount).toString()
                    else -> "E"
                } + when (it.owner) {
                    null -> "N"
                    else -> ((gameManager.playerCount + it.owner!!.index - player.index) % gameManager.playerCount).toString()
                }
            })
        }
    }

    private fun applyRules(turn: Int) {
        board = Board2D(board.width, board.height) { x, y ->
            var counter = 0
            val playerCounters = mutableMapOf<Player, Int>()
            for (xd in -1..1) {
                for (yd in -1..1) {
                    if (0 != xd || 0 != yd) {
                        val targetX = (x + xd + board.width) % board.width
                        val targetY = (y + yd + board.height) % board.height
                        val field = board[targetX, targetY]
                        if (null != field.village) {
                            counter++
                            if (field.village is PlayerVillage) {
                                playerCounters[(field.village as PlayerVillage).owner] =
                                    1 + (playerCounters[(field.village as PlayerVillage).owner] ?: 0)
                            }
                        }
                    }
                }
            }
            var max = 0
            var mostByPlayer: Player? = null
            playerCounters.forEach { (player, playerCounter) ->
                if (playerCounter > max) {
                    max = playerCounter
                    mostByPlayer = player
                } else if (playerCounter == max) {
                    mostByPlayer = null
                }
            }
            val newField = board[x, y].copy()
            if (null == board[x, y].village && 3 == counter) {
                newField.village =
                    if (null != mostByPlayer)
                        PlayerVillage(mostByPlayer!!, turn)
                    else
                        NeutralVillage(turn)
            } else if (null !== board[x, y].village) {
                if (1 >= counter || 4 <= counter) {
                    newField.village = null
                }
            }
            newField
        }
    }

    private fun capture() {
        for ((x, y) in board) {
            if (board[x, y].village is PlayerVillage) {
                board[x, y].owner = (board[x, y].village as PlayerVillage).owner
            }
        }
    }

    private fun redraw() {
        var neutralExists = false
        for ((x, y) in board) {
            val field = board[x, y]
            if (null == field.owner) {
                neutralExists = true
            }
            field.sprite!!.image =
                if (null == field.village)
                    "empty.png"
                else
                    "home.png"
            field.sprite!!.tint =
                if (null == field.owner)
                    if (null != field.village)
                        0xffffff
                    else
                        0x000000
                else
                    field.owner!!.colorToken
            field.sprite!!.alpha =
                if (field.village is PlayerVillage)
                    1.0
                else
                    0.5
            val villageCreation = when (field.village) {
                is Village -> "\nVillage anno ${field.village!!.creation}"
                else -> ""
            }
            tooltipModule.setTooltipText(field.sprite, "($x, $y)$villageCreation")
        }

        gameManager.players.forEach {
            var villages = 0
            var owned = 0
            board.forEach { (x, y) ->
                val field = board[x, y]
                if (it == field.owner) {
                    owned++
                }
                if (field.village is PlayerVillage && it == (field.village as PlayerVillage).owner) {
                    villages++
                }
            }
            it.score = owned * (1 + board.width * board.height) + villages
            it.scoreText.text = it.score.toString()
            it.statsText.text = "$villages / $owned"
            tooltipModule.setTooltipText(
                it.scoreText, "ownedCells * (1 + width * height) + villages\n" +
                        "$owned * (1 + ${board.width} * ${board.height}) + $villages"
            )
            if (it.score == 0 && !neutralExists && it.isActive) {
                it.deactivate("${it.nicknameToken}: Driven out of the grid")
            }
        }

        val userOffsetY = (
                CANVAS_HEIGHT
                        - gameManager.players.size * (USER_CONTAINER_HEIGHT + USER_MARGIN_BOTTOM)
                        + USER_MARGIN_BOTTOM) / 2
        gameManager.players.sortedByDescending {
            it.score
        }.forEachIndexed { index, player ->
            player.group.y = userOffsetY + index * (USER_CONTAINER_HEIGHT + USER_MARGIN_BOTTOM)
        }
    }

    private fun initBackground() {
        graphicEntityModule.createSprite()
            .setImage("Background.jpg")
            .setAnchor(0.0)
        graphicEntityModule.createSprite()
            .setImage("logoCG.png")
            .setX(CANVAS_WIDTH - LOGO_OFFSET)
            .setY(CANVAS_HEIGHT - LOGO_OFFSET)
            .setAnchor(1.0)
    }
}