package com.codingame.game;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.lang.Math;

import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.GameManager;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.endscreen.EndScreenModule;
import com.codingame.gameengine.module.entities.*;
import com.codingame.gameengine.module.tooltip.TooltipModule;
import com.google.inject.Inject;
import com.google.inject.Provider;

public class Referee extends AbstractReferee {
    @Inject private MultiplayerGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;
    @Inject private EndScreenModule endScreenModule;
    @Inject private TooltipModule tooltips;



    private double pi_2=Math.PI/2;
    private int dimension;
    private int[][][] grilles=new int[2][][]; // 0 si vide, 1 pour joueur 0 et 2 pour joueur 1
    private Rectangle[][][] grilles_graphique=new Rectangle[2][][];
    private int[][] fourmis=new int[2][] ; // (row,col,dir,joueur)  dir = 0 à l'est puis sens trigo
    private Polygon[] fourmis_graphique=new Polygon[2];
    private Action lastAction = null;
    private int nb_cases_a_placer; // Nombre de cases que chaque joueur doit colorier
    private int nb_tours_fourmi; // Nombre de pas que va faire la fourmi
    private int nb_grilles_differentes;
    // Traduction direction -> modificateur x et y
    private int[] dir_col = {1, 0, -1, 0};
    private int[] dir_row = {0, -1, 0, 1};

    private int dim_cases;
    private int[] origines_x;
    private int[] origines_y;
    private boolean[] touche_bord=new boolean[]{false,false};

    private Text[] scores=new Text[2];
    private int[] scores_partiels=new int[4];
    private int longueur_round;


    @Override
    public void init() {
        Random random = new Random(gameManager.getSeed());

        gameManager.setFrameDuration(400);

        if (gameManager.getLeagueLevel() >=3) {
            nb_grilles_differentes=1;
            dimension=25+random.nextInt(11);
            nb_cases_a_placer=15+random.nextInt(10);
            nb_tours_fourmi=100+2*random.nextInt(50);


        } else if(gameManager.getLeagueLevel() ==1) {
            nb_grilles_differentes=2;
            dimension=15;
            nb_cases_a_placer=20;
            nb_tours_fourmi=150;

        } else if(gameManager.getLeagueLevel() ==2) {
            nb_grilles_differentes=2;
            dimension=10+random.nextInt(10);
            nb_cases_a_placer=15+random.nextInt(25);
            nb_tours_fourmi=100+2*random.nextInt(50);
        }

        if(nb_grilles_differentes==2){
            dim_cases=(500/dimension);
            grilles=new int[2][dimension][dimension] ;
            grilles_graphique=new Rectangle[2][dimension][dimension] ;
            origines_x=new int[]{920-dimension*dim_cases,1000};
            origines_y=new int[]{540-dimension*dim_cases/2,540-dimension*dim_cases/2};
            fourmis = new int[][]{{dimension / 2, dimension / 2, 1, 0},{dimension / 2, dimension / 2, 1, 1}};
            fourmis_graphique=new Polygon[2];

        }else{
            dim_cases=(1000/dimension);
            int[][] grille=new int[dimension][dimension] ;
            Rectangle[][] grille_graphique=new Rectangle[dimension][dimension] ;
            grilles[0]=grilles[1]=grille;
            grilles_graphique[0]=grilles_graphique[1]=grille_graphique;
            origines_x=new int[]{960-dimension*dim_cases/2,960-dimension*dim_cases/2};
            origines_y=new int[]{540-dimension*dim_cases/2,540-dimension*dim_cases/2};
            fourmis = new int[][]{{dimension / 2, dimension / 2, 1, 0},{dimension / 2, dimension / 2, 1, 0}};
            fourmis_graphique=new Polygon[2];
        }

        longueur_round=2*nb_cases_a_placer+nb_tours_fourmi;
        gameManager.setMaxTurns(longueur_round*2+3);
        gameManager.setTurnMaxTime(300);

        drawBackground();
        drawHud();
        drawGrids();
    }

    private void reinit(){ // Réinitialse tout à la fin du premier round
        lastAction=null;
        touche_bord=new boolean[]{false,false};

        gameManager.setFrameDuration(400);
        // On sauvegarde les scores et reinitialise
        scores_partiels=new int[]{gameManager.getPlayer(0).getScore(),gameManager.getPlayer(1).getScore(),0,0};
        gameManager.getPlayer(0).setScore(0);gameManager.getPlayer(1).setScore(0);
        // On reset la grille
        int[][] grille=new int[dimension][dimension] ;
        grilles[0]=grilles[1]=grille;
        for (int n=0;n<nb_grilles_differentes;n++) {
            for (int i = 0; i < dimension; i++) {
                for (int j = 0; j < dimension; j++) {
                    grilles_graphique[n][i][j].setFillColor(0xffffff);
                }
            }
        }
        // On reset la fourmi
        fourmis = new int[][]{{dimension / 2, dimension / 2, 1, 1},{dimension / 2, dimension / 2, 1, 1}};
        for (int n=0;n<nb_grilles_differentes;n++) {
            fourmis_graphique[n].setX(origines_x[n]+fourmis[n][1]*dim_cases+dim_cases/2)
                    .setY(origines_y[n]+fourmis[n][0]*dim_cases+dim_cases/2)
                    .setFillColor(gameManager.getPlayer(1).getColorToken())
                    .setRotation(-pi_2*fourmis[n][2]);
        }

    }

    private void drawBackground() {
        graphicEntityModule.createSprite()
                .setImage("Background.jpg")
                .setAnchor(0);

    }

    private void drawGrids() {

        // Dessin des cases
        for (int n=0;n<nb_grilles_differentes;n++) {
            for (int i = 0; i < dimension; i++) {
                for (int j = 0; j < dimension; j++) {

                    grilles_graphique[n][i][j] = graphicEntityModule
                            .createRectangle()
                            .setWidth(dim_cases)
                            .setHeight(dim_cases)
                            .setX(origines_x[n] + j * dim_cases)
                            .setY(origines_y[n] + i * dim_cases)
                            .setLineWidth(1)
                            .setLineColor(0xc8c8c8)
                            .setFillColor(0xffffff);

                    // Le tooltip associé à la case
                    tooltips.setTooltipText(grilles_graphique[n][i][j], String.format("( %d ; %d)",i,j));

                }
            }
        // Fourmi triangulaire :
        fourmis_graphique[n]=graphicEntityModule
                .createPolygon()
                .setX(origines_x[n]+fourmis[n][1]*dim_cases+dim_cases/2)
                .setY(origines_y[n]+fourmis[n][0]*dim_cases+dim_cases/2)
                .addPoint(3*dim_cases/8,0)
                .addPoint(-3*dim_cases/8,2*dim_cases/8)
                .addPoint(-3*dim_cases/8,-2*dim_cases/8)
                .addPoint(3*dim_cases/8,0)
                .setLineWidth(1)
                .setLineColor(0x000000)
                .setFillColor(gameManager.getPlayer(0).getColorToken())
                .setRotation(-pi_2*fourmis[n][2]);
        }

        if (nb_grilles_differentes==2){ //On rajoute un séparateur
            graphicEntityModule.createLine()
                    .setX(960)
                    .setY(1)
                    .setX2(960)
                    .setY2(1079)
                    .setLineWidth(3)
                    .setLineColor(0xff0000);
        }
    }

    private void maj_couleur_case(int row,int col,int couleur,int numero){
        grilles_graphique[numero][row][col].setFillColor(0x000000, Curve.ELASTIC);
        grilles_graphique[numero][row][col].setFillColor(couleur, Curve.ELASTIC);
    }


    private void drawHud() {
        for (Player player : gameManager.getPlayers()) {
            int x = player.getIndex() == 0 ? 200 : 1920 - 200;
            int y = 220;

            graphicEntityModule
                    .createRectangle()
                    .setWidth(140)
                    .setHeight(140)
                    .setX(x - 70)
                    .setY(y - 70)
                    .setLineWidth(0)
                    .setFillColor(player.getColorToken());

            graphicEntityModule
                    .createRectangle()
                    .setWidth(120)
                    .setHeight(120)
                    .setX(x - 60)
                    .setY(y - 60)
                    .setLineWidth(0)
                    .setFillColor(0xffffff);

            Text text = graphicEntityModule.createText(player.getNicknameToken())
                    .setX(x)
                    .setY(y + 120)
                    .setZIndex(20)
                    .setFontSize(40)
                    .setFillColor(0xffffff)
                    .setAnchor(0.5);

            Sprite avatar = graphicEntityModule.createSprite()
                    .setX(x)
                    .setY(y)
                    .setZIndex(20)
                    .setImage(player.getAvatarToken())
                    .setAnchor(0.5)
                    .setBaseHeight(116)
                    .setBaseWidth(116);

            Text score = graphicEntityModule.createText(Integer.toString(player.getScore()))
                    .setX(x)
                    .setY(y + 240)
                    .setZIndex(20)
                    .setFontSize(80)
                    .setFillColor(0xffffff)
                    .setAnchor(0.5);
            scores[player.getIndex()]=score;

            player.hud = graphicEntityModule.createGroup(text, avatar);
        }
    }

    private void sendInputs(Player player) {
        // last action
        if (lastAction != null) {
            player.sendInputLine(lastAction.toString());
        } else {
            player.sendInputLine("-1 -1");
        }
    }

    private void setWinner(boolean fini) {
        if (fini) { //fin de la partie
            // On fait le total des points
            scores_partiels[2]=gameManager.getPlayer(0).getScore();scores_partiels[3]=gameManager.getPlayer(1).getScore();
            gameManager.getPlayer(0).setScore(gameManager.getPlayer(0).getScore()+scores_partiels[0]);
            gameManager.getPlayer(1).setScore(gameManager.getPlayer(1).getScore()+scores_partiels[1]);

            if (gameManager.getPlayer(0).getScore() < gameManager.getPlayer(1).getScore()) {
                gameManager.addTooltip(gameManager.getPlayer(1), gameManager.getPlayer(1).getNicknameToken() + " won the game!");
                gameManager.addToGameSummary(GameManager.formatSuccessMessage(gameManager.getPlayer(1).getNicknameToken() + " won the game!"));
            } else if (gameManager.getPlayer(0).getScore() > gameManager.getPlayer(1).getScore()) {
                gameManager.addTooltip(gameManager.getPlayer(0), gameManager.getPlayer(0).getNicknameToken() + " won the game!");
                gameManager.addToGameSummary(GameManager.formatSuccessMessage(gameManager.getPlayer(0).getNicknameToken() + " won the game!"));
            } else {
                gameManager.addTooltip(gameManager.getPlayer(0), " Draw!");
                gameManager.addToGameSummary(GameManager.formatSuccessMessage("Draw !"));
            }
            endGame();
        } else { // Fin de round seulement mais pas de la partie
            if (gameManager.getPlayer(0).getScore() < gameManager.getPlayer(1).getScore()) {
                gameManager.addTooltip(gameManager.getPlayer(1), gameManager.getPlayer(1).getNicknameToken() + " won this round!");
                gameManager.addToGameSummary(GameManager.formatSuccessMessage(gameManager.getPlayer(1).getNicknameToken() + " won this round!"));
            } else if (gameManager.getPlayer(0).getScore() > gameManager.getPlayer(1).getScore()) {
                gameManager.addTooltip(gameManager.getPlayer(0), gameManager.getPlayer(0).getNicknameToken() + " won this round!");
                gameManager.addToGameSummary(GameManager.formatSuccessMessage(gameManager.getPlayer(0).getNicknameToken() + " won this round!"));
            } else {
                gameManager.addTooltip(gameManager.getPlayer(0), " Draw!");
                gameManager.addToGameSummary(GameManager.formatSuccessMessage("Draw !"));
            }
        }
    }

    private void majScores(){
        //MAJ scores
        scores[0].setText(Integer.toString(gameManager.getPlayer(0).getScore()));
        scores[1].setText(Integer.toString(gameManager.getPlayer(1).getScore()));
    }

    @Override
    public void gameTurn(int turn) {
        Player player = gameManager.getPlayer(turn % 2);
        int indice=turn % nb_grilles_differentes;
        int[][] grille= grilles[indice];
        int[] fourmi= fourmis[indice];
        Polygon fourmi_graphique=fourmis_graphique[indice];
        int turn_bis=(turn-turn/longueur_round)%longueur_round; // Pour ramener le tour du joueur modulo longueur_round et que 0 corresponde au premier joueur

        if (turn_bis==2*nb_cases_a_placer){// Quand c'est à la fourmi de bouger on accélère l'affichage
            gameManager.setFrameDuration(250);
        }

        if(turn==longueur_round*2+1) {// Si c'est fini
            setWinner(false);
        }else if(turn==longueur_round*2+2){
            setWinner(true);
        }else if (turn==longueur_round) {// Si le premier round est fini, on reinitialise tout, on envoie -2 au premier joueur
            if(nb_grilles_differentes==2){ // Si on joue seul, pas besoin de second tour
                setWinner(true);
            } else {
                setWinner(false);
                reinit();
                player.sendInputLine("-2 -2");
                player.execute();
                try{
                    final Action action = player.getAction();
                }catch (Exception e){

                }
            }

        }else if (turn_bis<nb_cases_a_placer*2 ) { // Si on est dans la phase de choix des joueurs
            if (turn<2){
                player.sendInputLine(Integer.toString(dimension));
                player.sendInputLine(Integer.toString(nb_cases_a_placer));
                player.sendInputLine(Integer.toString(nb_tours_fourmi/nb_grilles_differentes));
            }
            sendInputs(player);
            player.execute();

            // Read inputs
            try {
                final Action action = player.getAction();
                gameManager.addToGameSummary(String.format("Player %s played (%d %d)", action.player.getNicknameToken(), action.row, action.col));

                if (grille[action.row][action.col]!=0) {
                    throw new InvalidAction("Invalid action.");
                }
                lastAction = action;
                grille[action.row][action.col]=action.player.getIndex()+1;
                player.setScore(player.getScore() + 1);
                maj_couleur_case(action.row,action.col,action.player.getColorToken(),action.player.getIndex()%nb_grilles_differentes);
                majScores();

            } catch (NumberFormatException e) {
                gameManager.addToGameSummary(GameManager.formatErrorMessage(player.getNicknameToken() + " wrong output!"));
                player.deactivate("Wrong output!");
                player.setScore(-1);
                endGame();
            } catch (TimeoutException e) {
                gameManager.addToGameSummary(GameManager.formatErrorMessage(player.getNicknameToken() + " timeout!"));
                player.deactivate(player.getNicknameToken() + " timeout!");
                player.setScore(-1);
                endGame();
            } catch (InvalidAction e) {
                gameManager.addToGameSummary(GameManager.formatErrorMessage(player.getNicknameToken() + " invalid action"));
                player.deactivate(e.getMessage());
                player.setScore(-1);
                endGame();
            }catch (ArrayIndexOutOfBoundsException e) {
                gameManager.addToGameSummary(GameManager.formatErrorMessage(player.getNicknameToken() + " invalid action"));
                player.deactivate(e.getMessage());
                player.setScore(-1);
                endGame();
            }

        }  else if(!touche_bord[turn%nb_grilles_differentes]){ // Si on est dans la phase où la fourmi avance sans avoir touché le bord
            if (turn_bis==nb_cases_a_placer*2 ) {// Si on est au premier tour de la fourmi, on ajoute une marque dans la barre de progression du la fenetre graphique
                gameManager.addTooltip(player,"The ant start moving");
            }


            // On change la couleur de la case où elle est
            if (grille[fourmi[0]][fourmi[1]]==0){// Si la case est vide
                fourmi[2]=(fourmi[2]+3)%4;// On fait tourner la fourmi
                fourmi_graphique.setRotation(-pi_2*fourmi[2],Curve.LINEAR);
                grille[fourmi[0]][fourmi[1]]=fourmi[3]+1; // ! décalage du numéro du joueur dans la grille
                maj_couleur_case(fourmi[0],fourmi[1],gameManager.getPlayer(fourmi[3]).getColorToken(),gameManager.getPlayer(fourmi[3]).getIndex()%nb_grilles_differentes); // On maj la couleur
                gameManager.getPlayer(fourmi[3]).setScore(gameManager.getPlayer(fourmi[3]).getScore()+1); // On maj le score
            } else{ // Sinon on enleve la couleur
                fourmi[2]=(fourmi[2]+1)%4;// On fait tourner la fourmi
                fourmi_graphique.setRotation(-pi_2*fourmi[2],Curve.LINEAR);
                gameManager.getPlayer(grille[fourmi[0]][fourmi[1]]-1).setScore(gameManager.getPlayer(grille[fourmi[0]][fourmi[1]]-1).getScore()-1); // maj score
                fourmi[3]=grille[fourmi[0]][fourmi[1]]-1; // Maj numero du joueur associé à la fourmi
                fourmi_graphique.setFillColor(gameManager.getPlayer(fourmi[3]).getColorToken(),Curve.ELASTIC); // maj couleur foumi
                grille[fourmi[0]][fourmi[1]]=0; // maj grille
                maj_couleur_case(fourmi[0],fourmi[1],0xffffff,gameManager.getPlayer(fourmi[3]).getIndex()%nb_grilles_differentes); // maj couleur case

            }
            // On fait avancer la fourmi
            fourmi[0]+=dir_row[fourmi[2]];
            fourmi[1]+=dir_col[fourmi[2]];
            if (fourmi[0]<dimension && fourmi[0]>=0 && fourmi[1]<dimension && fourmi[1]>=0){
                fourmi_graphique.setX(origines_x[indice]+fourmi[1]*dim_cases+dim_cases/2,Curve.LINEAR)
                        .setY(origines_y[indice]+fourmi[0]*dim_cases+dim_cases/2);

            } else{
                gameManager.addToGameSummary("The ant go out of the grid !");
                touche_bord[turn%nb_grilles_differentes]=true;
            }
            majScores();
        } else {
            gameManager.addToGameSummary("The ant go out of the grid !");
        }
    }

    private void endGame() {
        gameManager.endGame();

        Player p0 = gameManager.getPlayers().get(0);
        Player p1 = gameManager.getPlayers().get(1);
        if (p0.getScore() > p1.getScore()) {
            p1.hud.setAlpha(0.3);
        }
        if (p0.getScore() < p1.getScore()) {
            p0.hud.setAlpha(0.3);
        }

    }

    @Override
    public void onEnd() {
        if (nb_grilles_differentes==1) {
            int[] scores = {gameManager.getPlayer(0).getScore(), gameManager.getPlayer(1).getScore()};
            String[] text = {String.format("%d + %d = %d", scores_partiels[0], scores_partiels[2], scores[0]), String.format("%d + %d = %d", scores_partiels[1], scores_partiels[3], scores[1])};
            endScreenModule.setScores(scores, text);
        } else{
            int[] scores = {gameManager.getPlayer(0).getScore(), gameManager.getPlayer(1).getScore()};
            endScreenModule.setScores(scores);
        }
    }
}
