package com.codingame.game;
import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.entities.Sprite;
import com.codingame.gameengine.module.entities.Text;
import com.codingame.gameengine.module.entities.Line;
import com.google.inject.Inject;
import java.lang.Math; 

public class Referee extends AbstractReferee {
    // Uncomment the line below and comment the line under it to create a Solo Game
    // @Inject private SoloGameManager<Player> gameManager;
    @Inject private MultiplayerGameManager<Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;
    
    //CG config variables
    private int GAMETURN_LIMIT = 300;
    private int MAX_TURN_TIME = 50;
    private int game_seed;
    		
    //Game map
    //X = Space , . = Air, # = Dirt
    private int mapSize = 33;
    private char[][] map = new char[mapSize][mapSize];

    //Player info
    private int num_of_players = 2;
    private class Worm_owner
    {
    	public int id;
    	public int score;
    	public int next_worm_id;
    };
    private Worm_owner[] worm_owner_list = new Worm_owner[num_of_players];
    
    //Worm info
    private int num_of_worms = 6;
    private int max_health = 150;
    private int move_range = 1;
    private int dig_range = 1;
    private int weapon_damage = 8;
    private int weapon_range = 4;
    
    private class Worm
    {
    	public int id;
    	public int owner;
    	public int health;
    	public int row;
    	public int col;
    	public int dig_range;
    	public int move_range;
    	public int weapon_damage;
    	public int weapon_range;
    	
    };
    private Worm[] worm_list = new Worm[num_of_worms];
    
    //Helathpack info
    private int num_of_health_packs = 2;
    private int healing_strength = 10;
    
    class Potion
    {
    	public int id;
    	public String type;
    	public int strength;
    	public int row;
    	public int col;
    };
    private Potion[] potion_list = new Potion[num_of_health_packs];
    
    //Map Generator config
    private double mapZoom = 0.3;
    private double amountOfDirt = 0.5;
    private double mapCenter = (mapSize - 1) / 2;
    private double mapRadiusFit = mapCenter;
    private int wormSpawnDistanceFromEdge = 1;
    
    //Game scoring config
    private int enemy_worm_kill = 40;
    private int enemy_worm_hit = 20;
    private int own_worm_hit = -20;
    private int missed_attack = 2;
    private int move_score = 5;
    private int dig_score = 7;
    private int invalid_command_score = -4;
    
    //collisiong config
    private int collision_health = -20;
    
    //Map Graphics config
    private int map_top = 10;
    private int map_left = 10;
    private int mapcell_width = 32;
    private int mapcell_height = 32;
    
    //Map Graphics
    private Sprite[][] map_sprite = new Sprite[mapSize][mapSize];
    
    //Potion Graphics
    private Sprite[] potion_sprite = new Sprite[num_of_health_packs];
    private Sprite[] potion_mask = new Sprite[num_of_health_packs];
    
    //Worm graphics
    private Sprite[] worm_sprite = new Sprite[num_of_worms];
    private Sprite[] worm_mask = new Sprite[num_of_worms];
    
    //Worm Healthbar graphics
    private Line[][] health_bars = new Line[num_of_worms][2];
    
    //Player Graphics config
    private int player_top = 10;
    private int player_left = 1080;
    private int player_height = 500;
    
    //Player Graphics
    //private Sprite[] player_sprite = new Sprite[num_of_players];
    private Text[] player_score  = new Text[num_of_players];
    
    
    //Shooting Lazer Graphics
    private Line[] shooting_lazers = new Line[4];
    
    @Override
    public void init() 
    {
        gameManager.setMaxTurns(GAMETURN_LIMIT);
        gameManager.setTurnMaxTime(MAX_TURN_TIME);
        
        graphicEntityModule.createSprite()
                .setImage("background.jpg")
                .setX(0)
        		.setY(0)
                .setAnchor(0);
        
        game_seed = (int)gameManager.getSeed();
        
        generate_start_state();
    }

    @Override
    public void gameTurn(int turn) 
    {
    	String[] player_move_list = new String[num_of_players];
        for(int player_index = 0;player_index<num_of_players;player_index++) 
        {
            Player player = gameManager.getPlayer(player_index);
            
            sendPlayerInputs(player,turn,player_index);
            
            player.execute();
            
        }
        
        for(int player_index = 0;player_index<num_of_players;player_index++) 
        {
        	Player player = gameManager.getPlayer(player_index);
        	
            try 
            {
                //get input from player
                String player_move = player.getOutputs().get(0);
                
                // Check validity of the player output and compute the new game state
                String move_error = is_move_legal(player_move);
                if(move_error != "")
                {
                	player.deactivate(move_error);
                	player_move_list[player_index] = "";
	        		gameManager.endGame();
                }
                else
                {
                	player_move_list[player_index] = player_move;
                }
                
            } 
            catch (TimeoutException e) 
            {
                player.deactivate("Time limit exceeded!");
                gameManager.endGame();
                return;
            }
        }
        
        //process moves to game state here
        
        //remove previous moves shooting
        reset_shoot_alpha();
        
        apply_move(player_move_list[0] , player_move_list[1]);
        apply_dig(player_move_list[0] , player_move_list[1]);
        apply_shoot(player_move_list[0] , player_move_list[1] , turn);
        
        update_next_worm_id();
        
        update_graphics();
        
        check_win_condition();
    }
    
    @Override
    public void onEnd() 
    {
    	int[] score = new int[num_of_players];
    	
    	for(int i=0;i<num_of_players;i++)
    	{
    		if(gameManager.getPlayer(i).isActive())
    		{
    			score[i] = 1000000;
    		}
    		score[i] += worm_owner_list[i].score;
    		
    		gameManager.getPlayer(i).setScore(score[i]);
    	}
    }
    
    private void generate_start_state()
    {
    	populate_cells();
    	setup_worm_owners();
    	setup_worms();
    	setup_walls_around_worms();
    	set_battle_map_edges();
    	place_power_ups();
    			
    	build_map_graphics();
    	build_potion_graphics();
    	build_worm_graphics();
    	build_worm_healthbars();
    	build_avatar_graphics();
    	build_shoot_graphics();
    }
    
    private void populate_cells()
    {
    	for(int row = 0;row<mapSize;row++)
    	{
    		for(int col = 0;col<mapSize;col++)
    		{
    			if(col > (mapSize - 1) / 2)
    			{
    				double cell_type = n2d(((mapSize - 1) - col) * mapZoom , row * mapZoom, (int)game_seed);
    				if(cell_type > amountOfDirt)
    				{
    					map[row][col] = '.';
    				}
    				else
    				{
    					map[row][col] = '#';
    				}
    			}
    			else
    			{
    				double cell_type = n2d(col * mapZoom , row * mapZoom, (int)game_seed);
    				if(cell_type > amountOfDirt)
    				{
    					map[row][col] = '.';
    				}
    				else
    				{
    					map[row][col] = '#';
    				}
    			}
    		}
    	}

    }
    
    private double n2d(double xin,double yin, int seed)
    {
    	int[] p = {151, 160, 137, 91, 90, 15, 131, 13, 201, 95, 96, 53, 194, 233, 7, 225, 140, 36, 103,
    		30, 69, 142, 8, 99, 37, 240, 21, 10, 23, 190, 6, 148, 247, 120, 234, 75, 0, 26, 197, 62, 94, 252, 219, 203,
    		117, 35, 11, 32, 57, 177, 33, 88, 237, 149, 56, 87, 174, 20, 125, 136, 171, 168, 68, 175, 74, 165, 71, 134,
    		139, 48, 27, 166, 77, 146, 158, 231, 83, 111, 229, 122, 60, 211, 133, 230, 220, 105, 92, 41, 55, 46, 245,
    		40, 244, 102, 143, 54, 65, 25, 63, 161, 1, 216, 80, 73, 209, 76, 132, 187, 208, 89, 18, 169, 200, 196, 135,
    		130, 116, 188, 159, 86, 164, 100, 109, 198, 173, 186, 3, 64, 52, 217, 226, 250, 124, 123, 5, 202, 38, 147,
    		118, 126, 255, 82, 85, 212, 207, 206, 59, 227, 47, 16, 58, 17, 182, 189, 28, 42, 223, 183, 170, 213, 119,
    		248, 152, 2, 44, 154, 163, 70, 221, 153, 101, 155, 167, 43, 172, 9, 129, 22, 39, 253, 19, 98, 108, 110, 79,
    		113, 224, 232, 178, 185, 112, 104, 218, 246, 97, 228, 251, 34, 242, 193, 238, 210, 144, 12, 191, 179, 162,
    		241, 81, 51, 145, 235, 249, 14, 239, 107, 49, 192, 214, 31, 181, 199, 106, 157, 184, 84, 204, 176, 115, 121,
    		50, 45, 127, 4, 150, 254, 138, 236, 205, 93, 222, 114, 67, 29, 24, 72, 243, 141, 128, 195, 78, 66, 215, 61,
    		156, 180};
    	int[] perm = new int [512];
    	int[] permMod12 = new int [512];
    	
    	for(int i = 0;i<512;i++)
    	{
    		perm[i] = p[i&255];
    		permMod12[i] = perm[i] %12;
    	}
    	
    	double[][] grad3 =
    		{
    				{1.0, 1.0, 0.0  },
    				{-1.0, 1.0, 0.0 },
    				{1.0, -1.0, 0.0 },
    				{-1.0, -1.0, 0.0},
    				{1.0, 0.0, 1.0  },
    				{-1.0, 0.0, 1.0 },
    				{1.0, 0.0, -1.0 },
    				{-1.0, 0.0, -1.0},
    				{0.0, 1.0, 1.0  },
    				{0.0, -1.0, 1.0 },
    				{0.0, 1.0, -1.0 },
    				{0.0, -1.0, -1.0}
    		};
    			
    			
    	double F3 = 1.0 / 3.0;
    	double G3 = 1.0 / 6.0;
    	
    	double n0,n1,n2,n3;
    	double s = (xin + yin + seed) * F3;
    	double i = Math.floor(xin + s);
    	double j = Math.floor(yin + s);
    	double k = Math.floor(seed + s);
    	double t = (i+j+k) * G3;
    	
    	double X0 = i - t;
    	double Y0 = j - t;
    	double Z0 = k - t;
    	
    	double x0 = xin - X0;
    	double y0 = yin - Y0;
    	double z0 = seed - Z0;
    	
    	int i1,j1,k1,i2,j2,k2;
    	
    	if (x0 >= y0) 
    	{
            if (y0 >= z0) 
            {
                i1 = 1;
                j1 = 0;
                k1 = 0;
                i2 = 1;
                j2 = 1;
                k2 = 0;
            }
            else if (x0 >= z0) 
            {
                i1 = 1;
                j1 = 0;
                k1 = 0;
                i2 = 1;
                j2 = 0;
                k2 = 1;
            }
            else 
            {
                i1 = 0;
                j1 = 0;
                k1 = 1;
                i2 = 1;
                j2 = 0;
                k2 = 1;
            }
        } 
    	else
    	{
            if (y0 < z0) 
            {
                i1 = 0;
                j1 = 0;
                k1 = 1;
                i2 = 0;
                j2 = 1;
                k2 = 1;
            }
            else if (x0 < z0) 
            {
                i1 = 0;
                j1 = 1;
                k1 = 0;
                i2 = 0;
                j2 = 1;
                k2 = 1;
            }
            else 
            {
                i1 = 0;
                j1 = 1;
                k1 = 0;
                i2 = 1;
                j2 = 1;
                k2 = 0;
            } 
        }
    	
    	double x1 = x0 - i1 + G3;
    	double y1 = y0 - j1 + G3;
    	double z1 = z0 - k1 + G3;
    	double x2 = x0 - i2 + 2.0 * G3;
    	double y2 = y0 - j2 + 2.0 * G3;
    	double z2 = z0 - k2 + 2.0 * G3;
    	double x3 = x0 - 1.0 + 3.0 * G3;
    	double y3 = y0 - 1.0 + 3.0 * G3;
    	double z3 = z0 - 1.0 + 3.0 * G3;    	
    	
    	int ii = (int)(i) & 255;
    	int jj = (int)(j) & 255;
    	int kk = (int)(k) & 255;
    	
    	int gi0 = permMod12[ii + perm[jj + perm[kk]]];
    	int gi1 = permMod12[ii + i1 + perm[jj + j1 + perm[kk + k1]]];
    	int gi2 = permMod12[ii + i2 + perm[jj + j2 + perm[kk + k2]]];
    	int gi3 = permMod12[ii + 1 + perm[jj + 1 + perm[kk + 1]]];
    	
    	double t0 = 0.6 - x0 * x0 - y0 * y0 - z0 * z0;
    	double t1 = 0.6 - x1 * x1 - y1 * y1 - z1 * z1;
    	double t2 = 0.6 - x2 * x2 - y2 * y2 - z2 * z2;
    	double t3 = 0.6 - x3 * x3 - y3 * y3 - z3 * z3;
    	
    	if (t0 < 0)
    	{
            n0 = 0.0;
    	}
        else 
        {
            t0 *= t0;
            n0 = t0 * t0 * dot(grad3[gi0][0],grad3[gi0][1],grad3[gi0][2], x0, y0, z0);
        }
    	
    	if (t1 < 0)
    	{
            n1 = 0.0;
    	}
        else 
        {
            t1 *= t1;
            n1 = t1 * t1 * dot(grad3[gi1][0],grad3[gi1][1],grad3[gi1][2], x1, y1, z1);
        }
    	
    	if (t2 < 0)
    	{
            n2 = 0.0;
    	}
        else 
        {
            t2 *= t2;
            n2 = t2 * t2 * dot(grad3[gi2][0],grad3[gi2][1],grad3[gi2][2], x2, y2, z2);
        }
    	
    	if (t3 < 0)
    	{
            n3 = 0.0;
    	}
        else 
        {
            t3 *= t3;
            n3 = t3 * t3 * dot(grad3[gi3][0],grad3[gi3][1],grad3[gi3][2], x3, y3, z3);
        }
    	
    	return ((32.0 * (n0 + n1 + n2 + n3)) + 0.8679777777778225) * 0.576051614;
    }
    
    private double dot(double g1,double g2,double g3,double x,double y,double z)
    {
    	return g1*x + g2*y + g3*z;
    }
    
    private void setup_worm_owners()
    {
    	for(int i = 0 ;i < num_of_players;i++)
    	{
    		worm_owner_list[i] = new Worm_owner();
    		worm_owner_list[i].id = i;
    		worm_owner_list[i].score = 0;
    		worm_owner_list[i].next_worm_id = i;
    	}
    }
    
    private void setup_worms()
    {
    	for(int worm_id = 0;worm_id < num_of_worms;worm_id++)
    	{
    		worm_list[worm_id] = new Worm();
    		worm_list[worm_id].id = worm_id;
    		worm_list[worm_id].owner = worm_id % num_of_players;
    		worm_list[worm_id].health = max_health;
    		
    		double t = 2.0 * Math.PI * worm_id / num_of_worms;
    		worm_list[worm_id].col = (int)Math.round( (mapRadiusFit - wormSpawnDistanceFromEdge) * Math.cos(t) + mapCenter );
    		worm_list[worm_id].row = (int)Math.round( (mapRadiusFit - wormSpawnDistanceFromEdge) * Math.sin(t) + mapCenter );
    		
    		worm_list[worm_id].dig_range = dig_range;
    		worm_list[worm_id].move_range = move_range;
    		worm_list[worm_id].weapon_damage = weapon_damage;
    		worm_list[worm_id].weapon_range = weapon_range;
    	}
    	
    	
    }
    
    private void setup_walls_around_worms()
    {
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		int row = worm_list[i].row;
    		int col = worm_list[i].col;
    		
    		for(int delta_row = -2;delta_row<=2;delta_row++)
    		{
    			for(int delta_col = -2;delta_col<=2;delta_col++)
    			{
    				char target = '.';
    				if(Math.abs(delta_row) == 2 || Math.abs(delta_col) == 2)
    				{
    					target = '#';
    				}
    				
    				if(0 <= row + delta_row && row +delta_row < mapSize && 0<= col + delta_col && col+delta_col < mapSize)
    				{
    					map[row+delta_row][col + delta_col] = target;
    				}
    			}
    		}
    	}
    }
    
    private void set_battle_map_edges()
    {
    	for(int row = 0;row<mapSize;row++)
    	{
    		for(int col = 0;col<mapSize;col++)
    		{
    			int dist = (int)Math.sqrt( Math.pow(mapCenter - row,2) + Math.pow(mapCenter - col,2));
    			if(dist > mapRadiusFit)
    			{
    				map[row][col] = 'X';
    			}
    		}
    	}
    }
    
    private void place_power_ups()
    {
    	double tilt = n2d((double)num_of_health_packs,(double)0,(int)game_seed);
    	
    	for(int i = 0;i<num_of_health_packs;i++)
    	{
    		potion_list[i] = new Potion();
    		potion_list[i].type = "health";
    		potion_list[i].id = i;
    		potion_list[i].strength = healing_strength;
    		
    		double t = 2.0 * Math.PI * i / num_of_health_packs + tilt;
    		potion_list[i].row = (int)Math.round( 2.2 * Math.sin(t) + mapCenter );
    		potion_list[i].col = (int)Math.round( 2.2 * Math.cos(t) + mapCenter );
    		
    		map[potion_list[i].row][potion_list[i].col] = '.';
    	}
    }
    
    private void build_map_graphics()
    {
    	for(int row = 0;row<mapSize;row++)
    	{
    		for(int col = 0;col<mapSize;col++)
    		{
    			String image_file = "";
    			double alpha = 1;
    			if(map[row][col] == 'X')
    			{
    				image_file = "deep_space.png";
    				alpha = 0;
    			}
    			else if(map[row][col] == '.')
    			{
    				image_file = "air.png";
    			}
    			else if(map[row][col] == '#')
    			{
    				image_file = "dirt.png";
    			}
    			
    			map_sprite[row][col] = graphicEntityModule.createSprite()
    					.setImage(image_file)
    					.setX(map_left + col*mapcell_width)
    					.setY(map_top + row*mapcell_height)
    					.setBaseWidth(mapcell_width)
    					.setBaseHeight(mapcell_height)
    					.setAlpha(alpha)
    					;
    					
    		}
    	}
    }
    
    private void build_potion_graphics()
    {
    	for(int i = 0;i<num_of_health_packs;i++)
    	{
    		potion_mask[i] = graphicEntityModule.createSprite()
					.setImage("potion_mask.png")
					.setX(map_left + potion_list[i].col*mapcell_width)
					.setY(map_top + potion_list[i].row*mapcell_height)
					.setBaseWidth(mapcell_width)
					.setBaseHeight(mapcell_height)
					;
    		
    		
    		potion_sprite[i] = graphicEntityModule.createSprite()
					.setImage("potion.png")
					.setX(map_left + potion_list[i].col*mapcell_width)
					.setY(map_top + potion_list[i].row*mapcell_height)
					.setBaseWidth(mapcell_width)
					.setBaseHeight(mapcell_height)
					.setMask(potion_mask[i])
					.setZIndex(1)
					;
    	}
    }
    
    private void build_worm_graphics()
    {
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		String worm_image = "";
    		if(worm_list[i].owner == 0)
    		{
    			worm_image = "red_worm.png";
    		}
    		else if(worm_list[i].owner == 1)
    		{
    			worm_image = "blue_worm.png";
    		}
    		
    		worm_mask[i] = graphicEntityModule.createSprite()
					.setImage("worm_mask.png")
					.setX(map_left + worm_list[i].col*mapcell_width)
					.setY(map_top + worm_list[i].row*mapcell_height)
					.setBaseWidth(mapcell_width)
					.setBaseHeight(mapcell_height)
					;
    		
    		worm_sprite[i] = graphicEntityModule.createSprite()
					.setImage(worm_image)
					.setX(map_left + worm_list[i].col*mapcell_width)
					.setY(map_top + worm_list[i].row*mapcell_height)
					.setBaseWidth(mapcell_width)
					.setBaseHeight(mapcell_height)
					.setMask(worm_mask[i])
					.setZIndex(2)
					;
    	}
    }
    
    private void build_worm_healthbars()
    {
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		for(int f_line = 0;f_line <= 1;f_line++)
    		{
    			int r = Math.abs(f_line-1)*250;
    			int g = f_line*250;
    			int b = 0;
    			int rgb = 65536 * r + 256 * g + b;
    			
    			int line_length = mapcell_width;
    			if(f_line == 1)
    			{
    				line_length = worm_list[i].health * mapcell_width / max_health;
    			}
    			
    			health_bars[i][f_line] = graphicEntityModule.createLine()
    					.setLineWidth(1)
    					.setX(map_left + worm_list[i].col*mapcell_width)
    					.setY(map_top + worm_list[i].row*mapcell_height + mapcell_height)
    					.setX2(map_left + worm_list[i].col*mapcell_width + (line_length))
    					.setY2(map_top + worm_list[i].row*mapcell_height + mapcell_height)
    					.setLineColor(rgb)
    					.setZIndex(3 + f_line)
    					;
    		}
    	}
    }
    
    private void build_avatar_graphics()
    {
    	int r = 200;
    	int g = 200;
    	int b = 200;
    	
    	int text_col = 65536 * r + 256 * g + b;
    	
    	for(int i = 0;i<num_of_players;i++)
    	{
    		graphicEntityModule.createSprite()
    			.setImage(gameManager.getPlayer(i).getAvatarToken())
    			.setX(player_left)
    			.setY(player_top + i*player_height)
    			.setBaseWidth(200)
    			.setBaseHeight(200)
    			.setAnchor(0);
    		
    		graphicEntityModule.createText(gameManager.getPlayer(i).getNicknameToken())
    		    .setFillColor(text_col)
                .setFontSize(50)
                .setStrokeThickness(2)
                .setX(player_left + 15)
                .setY(player_top + i*player_height + 220)
                .setAnchor(0);
    		
    		graphicEntityModule.createText("Score:")
    		.setFillColor(text_col)
            .setFontSize(50)
            .setStrokeThickness(2)
            .setX(player_left + 15)
            .setY(player_top + i*player_height + 280)
            .setAnchor(0);
    		
    		player_score[i] = graphicEntityModule.createText( String.valueOf(worm_owner_list[i].score) )
    			.setFillColor(text_col)
                .setFontSize(50)
                .setStrokeThickness(2)
                .setX(player_left + 170)
                .setY(player_top + i*player_height + 280)
                .setAnchor(0);
    	}
    }
    
    private void build_shoot_graphics()
    {
    	for(int i = 0;i<4;i++)
    	{
    		shooting_lazers[i] = graphicEntityModule.createLine()
        			.setLineWidth(1)
        			.setX(0)
        			.setY(0)
        			.setX2(1)
        			.setY2(1)
        			.setLineColor(0)
        			.setZIndex(10)
        			.setAlpha(0);
    	}
    }
    
    private void sendPlayerInputs(Player player,int turn,int player_index)
    {
    	//initial input
        if(turn == 0)
        {
            player.sendInputLine(String.valueOf(player_index));
        }
        
        //Round number
        player.sendInputLine(String.valueOf(turn));
        
        //map size and map
        player.sendInputLine(String.valueOf(mapSize));
        for(int row = 0;row<mapSize;row++)
        {
        	String maprow = "";
        	for(int col = 0;col<mapSize;col++)
        	{
        		String nextchar = Character.toString(map[row][col]);
        		for(int i = 0;i<num_of_worms;i++)
        		{
        			if(worm_list[i].row == row && worm_list[i].col == col && worm_list[i].health > 0)
        			{
        				nextchar = String.valueOf(worm_list[i].owner);
        			}
        		}
        		
        		maprow = maprow + nextchar;
        	}
        	
        	player.sendInputLine(maprow);
        }
        
        //players info
        player.sendInputLine(String.valueOf(num_of_players));
        for(int player_num = 0;player_num<num_of_players;player_num++)
        {
        	String player_info = "";
        	player_info += String.valueOf(worm_owner_list[player_num].id);
        	player_info += " ";
        	player_info += String.valueOf(worm_owner_list[player_num].score);
        	player_info += " ";
        	player_info += String.valueOf(worm_owner_list[player_num].next_worm_id);
        	player.sendInputLine(player_info);
        }
        
        //worm info
        int worms_left = remaining_worms();
        player.sendInputLine(String.valueOf(worms_left));
        for(int worm_num = 0;worm_num<num_of_worms;worm_num++)
        {
        	if(worm_list[worm_num].health > 0)
        	{
        		String worm_info = "";
        		worm_info += String.valueOf(worm_list[worm_num].id);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].owner);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].health);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].col);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].row);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].dig_range);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].move_range);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].weapon_damage);worm_info += " ";
        		worm_info += String.valueOf(worm_list[worm_num].weapon_range);worm_info += " ";
        	
        		player.sendInputLine(worm_info);
        	}
        }
        
        //potioninfo
        int potions_left = remaining_potions();
        player.sendInputLine(String.valueOf(potions_left));
        for(int potion_num = 0;potion_num< num_of_health_packs;potion_num++)
        {
        	if(potion_list[potion_num].strength != 0)
        	{
        		String potion_info = "";
        		potion_info += String.valueOf(potion_list[potion_num].id);potion_info += " ";
        		potion_info += potion_list[potion_num].type;potion_info += " ";
        		potion_info += String.valueOf(potion_list[potion_num].strength);potion_info += " ";
        		potion_info += String.valueOf(potion_list[potion_num].col);potion_info += " ";
        		potion_info += String.valueOf(potion_list[potion_num].row);potion_info += " ";
        		
        		player.sendInputLine(potion_info);
        	}
        }
    }
    
    private int remaining_worms()
    {
    	int res = 0;
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		if(worm_list[i].health > 0)
    		{
    			res++;
    		}
    	}
    	return res;
    }
    
    private int remaining_potions()
    {
    	int res = 0;
    	for(int i = 0;i<num_of_health_packs;i++)
    	{
    		if(potion_list[i].strength != 0)
    		{
    			res++;
    		}
    	}
    	
    	return res;
    }
    
    private String is_move_legal(String player_move)
    {
    	String[] move_split = player_move.split(" ");
    	if(move_split.length == 0)
    	{
    		return "No legal command found, moves must start with the words nothing, move, dig or shoot";
    	}
    	else if(move_split[0].equals("nothing"))
    	{
    		if(move_split.length != 1)
    		{
    			return "nothing command must not include other variables";
    		}
    		else
    		{
    			return "";
    		}
    		
    	}
    	else if(move_split[0].equals("move"))
    	{
    		if(move_split.length != 3)
    		{
    			return "move commands must be int the format 'move row col'";
    		}
    		else if(move_split[1].matches("\\d+") && move_split[2].matches("\\d+"))
    		{
    			int row = Integer.parseInt(move_split[1]);
    			int col = Integer.parseInt(move_split[1]);
    			if(0<=row && row < mapSize && 0<= col && col < mapSize)
    			{
    				return "";
    			}
    			else
    			{
    				return "move commands must be int the format 'move row col' where row and col are integers between 0 and " + String.valueOf(mapSize-1);
    			}
    		}
    		else
    		{
    			return "move commands must be int the format 'move row col' where row and col are integers";
    		}
    	}
    	else if(move_split[0].equals("dig"))
    	{
    		if(move_split.length != 3)
    		{
    			return "dig commands must be in the format 'dig row col'";
    		}
    		else if(move_split[1].matches("\\d+") && move_split[2].matches("\\d+"))
    		{
    			int row = Integer.parseInt(move_split[1]);
    			int col = Integer.parseInt(move_split[1]);
    			if(0<=row && row < mapSize && 0<= col && col < mapSize)
    			{
    				return "";
    			}
    			else
    			{
    				return "dig commands must be int the format 'dig row col' where row and col are integers between 0 and " + String.valueOf(mapSize-1);
    			}
    		}
    		else
    		{
    			return "dig commands must be int the format 'dig row col' where row and col are integers";
    		}
    		
    	}
    	else if(move_split[0].equals("shoot"))
    	{
    		if(move_split.length != 2)
    		{
    			return "shoot commands must be in the format 'shoot dir'";
    		}
    		else if(move_split[1].equals("N") || move_split[1].equals("S") || move_split[1].equals("E") || move_split[1].equals("W") ||
    				move_split[1].equals("NE") || move_split[1].equals("NW") || move_split[1].equals("SE") || move_split[1].equals("SW"))
    		{
    			
    			return "";
    		}
    		else
    		{
    			return move_split[1] + "is not a direction";
    		}
    			
    	}
    	else
    	{
    		return "Command: " + player_move + " is not recognised. moves must start with the words nothing, move, dig or shoot";
    	}
    }
    
    private void apply_move(String player0_move,String player1_move)
    {
    	int valid_move0 = 1;
    	int valid_move1 = 1;
    	
    	String[] move_split0 = player0_move.split(" ");
    	String[] move_split1 = player1_move.split(" ");
    	
    	//check that move is not empty
    	if(valid_move0 == 1)
    	{
    		if(move_split0.length == 0)
    		{
    			valid_move0 = 0;
    		}	
    	}
    	if(valid_move1 == 1)
    	{
    		if(move_split1.length == 0)
    		{
    			valid_move1 = 0;
    		}	
    	}
    	
    	//check that it is a move command
    	if(valid_move0 == 1)
    	{
    		if(move_split0[0].equals("move") == false)
    		{
    			valid_move0 = 0;
    		}	
    	}
    	if(valid_move1 == 1)
    	{
    		if(move_split1[0].equals("move") == false)
    		{
    			valid_move1 = 0;
    		}	
    	}
    	
    	//check distance from worm
    	if(valid_move0 == 1)
    	{
    		int dist_of_move = dist(Integer.valueOf(move_split0[2]) , Integer.valueOf(move_split0[1]) ,
    					worm_list[worm_owner_list[0].next_worm_id].row, worm_list[worm_owner_list[0].next_worm_id].col);
    		if(dist_of_move > worm_list[worm_owner_list[0].next_worm_id].move_range || dist_of_move == 0)
    		{
    			valid_move0 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[0].score += invalid_command_score;
    		}
    	}
    	if(valid_move1 == 1)
    	{
    		int dist_of_move = dist(Integer.valueOf(move_split1[2]) , Integer.valueOf(move_split1[1]) ,
    					worm_list[worm_owner_list[1].next_worm_id].row, worm_list[worm_owner_list[1].next_worm_id].col);
    		if(dist_of_move > worm_list[worm_owner_list[1].next_worm_id].move_range  || dist_of_move == 0)
    		{
    			valid_move1 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[1].score += invalid_command_score;
    		}
    	}
    	
    	//check that target is empty and target is air
    	if(valid_move0 == 1)
    	{
    		//loop through worms
    		for(int i = 0;i<num_of_worms;i++)
    		{
    			if(i != worm_owner_list[0].next_worm_id && worm_list[i].health > 0)
    			{
    				if(worm_list[i].row == Integer.valueOf(move_split0[2]) && worm_list[i].col == Integer.valueOf(move_split0[1]))
    				{
    					//invalid move
    					valid_move0 = 0;
    	    			//move is invalid and so applied to score
    	    			worm_owner_list[0].score += invalid_command_score;
    				}
    			}
    		}
    		//check air
    		if(map[Integer.valueOf(move_split0[2])][Integer.valueOf(move_split0[1])] != '.')
    		{
    			//invalid move
    			valid_move0 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[0].score += invalid_command_score;
    		}
    	}
    	if(valid_move1 == 1)
    	{
    		//loop through worms
    		for(int i = 0;i<num_of_worms;i++)
    		{
    			if(i != worm_owner_list[1].next_worm_id && worm_list[i].health > 0)
    			{
    				if(worm_list[i].row == Integer.valueOf(move_split1[2]) && worm_list[i].col == Integer.valueOf(move_split1[1]))
    				{
    					//invalid move
    					valid_move1 = 0;
    	    			//move is invalid and so applied to score
    	    			worm_owner_list[1].score += invalid_command_score;
    				}
    			}
    		}
    		//check air
    		if(map[Integer.valueOf(move_split1[2])][Integer.valueOf(move_split1[1])] != '.')
    		{
    			//invalid move
    			valid_move1 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[1].score += invalid_command_score;
    		}
    	}
    	
    	//apply moves that are valid
    	if(valid_move0 == 1 && valid_move1 == 1)
    	{
    		//collision move
    		if( Integer.valueOf(move_split0[2]) == Integer.valueOf(move_split1[2])  && Integer.valueOf(move_split0[1]) == Integer.valueOf(move_split1[1]) )
    		{
    			double test = Math.random();
    			adjust_worm_health(worm_owner_list[0].next_worm_id,collision_health);
    			adjust_worm_health(worm_owner_list[1].next_worm_id,collision_health);
    			
    			if(test <0.5)
    			{
    				//swap worms
    				int temp_row = worm_list[worm_owner_list[0].next_worm_id].row;
    				int temp_col = worm_list[worm_owner_list[0].next_worm_id].col;
    				worm_list[worm_owner_list[0].next_worm_id].row = worm_list[worm_owner_list[1].next_worm_id].row;
    				worm_list[worm_owner_list[0].next_worm_id].col = worm_list[worm_owner_list[1].next_worm_id].col;
    				worm_list[worm_owner_list[1].next_worm_id].row = temp_row;
    				worm_list[worm_owner_list[1].next_worm_id].col = temp_col;
    			}
    		}
    		else
    		{
    			//usuall moves
    			move_worm(worm_owner_list[0].next_worm_id , Integer.valueOf(move_split0[2]) , Integer.valueOf(move_split0[1]));
    			move_worm(worm_owner_list[1].next_worm_id , Integer.valueOf(move_split1[2]) , Integer.valueOf(move_split1[1]));
    		}
    	}
    	else
    	{
    		//usuall moves
    		if(valid_move0 == 1)
    		{
    		    move_worm(worm_owner_list[0].next_worm_id , Integer.valueOf(move_split0[2]) , Integer.valueOf(move_split0[1]));
    		}
    		if(valid_move1 == 1)
    		{
    			move_worm(worm_owner_list[1].next_worm_id , Integer.valueOf(move_split1[2]) , Integer.valueOf(move_split1[1]));
    		}
    	}
    		
    }
    
    private void apply_dig(String player0_move,String player1_move)
    {
    	int valid_move0 = 1;
    	int valid_move1 = 1;
    	
    	String[] move_split0 = player0_move.split(" ");
    	String[] move_split1 = player1_move.split(" ");
    	
    	//check that move is not empty
    	if(valid_move0 == 1)
    	{
    		if(move_split0.length == 0)
    		{
    			valid_move0 = 0;
    		}	
    	}
    	if(valid_move1 == 1)
    	{
    		if(move_split1.length == 0)
    		{
    			valid_move1 = 0;
    		}	
    	}
    	
    	//check that it is a move command
    	if(valid_move0 == 1)
    	{
    		if(move_split0[0].equals("dig") == false)
    		{
    			valid_move0 = 0;
    		}	
    	}
    	if(valid_move1 == 1)
    	{
    		if(move_split1[0].equals("dig") == false)
    		{
    			valid_move1 = 0;
    		}	
    	}
    	
    	//check distance from worm
    	if(valid_move0 == 1)
    	{
    		int dist_of_move = dist(Integer.valueOf(move_split0[2]) , Integer.valueOf(move_split0[1]) ,
    					worm_list[worm_owner_list[0].next_worm_id].row, worm_list[worm_owner_list[0].next_worm_id].col);
    		if(dist_of_move > worm_list[worm_owner_list[0].next_worm_id].dig_range)
    		{
    			valid_move0 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[0].score += invalid_command_score;
    		}
    	}
    	if(valid_move1 == 1)
    	{
    		int dist_of_move = dist(Integer.valueOf(move_split1[2]) , Integer.valueOf(move_split1[1]) ,
    					worm_list[worm_owner_list[1].next_worm_id].row, worm_list[worm_owner_list[1].next_worm_id].col);
    		if(dist_of_move > worm_list[worm_owner_list[1].next_worm_id].dig_range)
    		{
    			valid_move1 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[1].score += invalid_command_score;
    		}
    	}
    	
    	//check that target is dirt
    	if(valid_move0 == 1)
    	{
    		//check dirt
    		if(map[Integer.valueOf(move_split0[2])][Integer.valueOf(move_split0[1])] != '#')
    		{
    			//invalid move
    			valid_move0 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[0].score += invalid_command_score;
    		}
    	}
    	if(valid_move1 == 1)
    	{
    		//check dirt
    		if(map[Integer.valueOf(move_split1[2])][Integer.valueOf(move_split1[1])] != '#')
    		{
    			//invalid move
    			valid_move1 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[0].score += invalid_command_score;
    		}
    	}
    	
    	if(valid_move0 == 1)
		{
		    dig_worm(worm_owner_list[0].next_worm_id , Integer.valueOf(move_split0[2]) , Integer.valueOf(move_split0[1]));
		}
		if(valid_move1 == 1)
		{
			dig_worm(worm_owner_list[1].next_worm_id , Integer.valueOf(move_split1[2]) , Integer.valueOf(move_split1[1]));
		}
    	
    }
    
    private void apply_shoot(String player0_move,String player1_move , int turn)
    {
    	int valid_move0 = 1;
    	int valid_move1 = 1;
    	
    	String[] move_split0 = player0_move.split(" ");
    	String[] move_split1 = player1_move.split(" ");
    	
    	//check that move is not empty
    	if(valid_move0 == 1)
    	{
    		if(move_split0.length == 0)
    		{
    			valid_move0 = 0;
    		}	
    	}
    	if(valid_move1 == 1)
    	{
    		if(move_split1.length == 0)
    		{
    			valid_move1 = 0;
    		}	
    	}

    	//check that it is a move command
    	if(valid_move0 == 1)
    	{
    		if(move_split0[0].equals("shoot") == false)
    		{
    			valid_move0 = 0;
    		}	
    	}
    	if(valid_move1 == 1)
    	{
    		if(move_split1[0].equals("shoot") == false)
    		{
    			valid_move1 = 0;
    		}	
    	}

    	//apply shooting
    	if(valid_move0 == 1)
    	{
    		int drow = 0;
    		int dcol = 0;
    		String dir = move_split0[1];
    		
    		if(dir.equals("N"))
    		{
    			drow = -1;
    		}
    		else if (dir.equals("NE"))
    		{
    			drow = -1;
    			dcol = 1;
    		}
    		else if (dir.equals("E"))
    		{
    			dcol = 1;
    		}
    		else if (dir.equals("SE"))
    		{
    			drow = 1;
    			dcol = 1;
    		}
    		else if (dir.equals("S"))
    		{
    			drow = 1;
    		}
    		else if (dir.equals("SW"))
    		{
    			drow = 1;
    			dcol = -1;
    		}
    		else if (dir.equals("W"))
    		{
    			dcol = -1;
    		}
    		else if (dir.equals("NW"))
    		{
    			drow = -1;
    			dcol = -1;
    		}
    		else
    		{
    			//invalid shooting direction
    			valid_move0 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[0].score += invalid_command_score;
    		}
    		
    		//apply shot here
    		if(valid_move0 == 1)
    		{
    			shoot_worm(worm_owner_list[0].next_worm_id,drow,dcol,turn);
    		}
    	}
    	
    	
    	if(valid_move1 == 1)
    	{
    		int drow = 0;
    		int dcol = 0;
    		String dir = move_split1[1];
    		
    		if(dir.equals("N"))
    		{
    			drow = -1;
    		}
    		else if (dir.equals("NE"))
    		{
    			drow = -1;
    			dcol = 1;
    		}
    		else if (dir.equals("E"))
    		{
    			dcol = 1;
    		}
    		else if (dir.equals("SE"))
    		{
    			drow = 1;
    			dcol = 1;
    		}
    		else if (dir.equals("S"))
    		{
    			drow = 1;
    		}
    		else if (dir.equals("SW"))
    		{
    			drow = 1;
    			dcol = -1;
    		}
    		else if (dir.equals("W"))
    		{
    			dcol = -1;
    		}
    		else if (dir.equals("NW"))
    		{
    			drow = -1;
    			dcol = -1;
    		}
    		else
    		{
    			//invalid shooting direction
    			valid_move1 = 0;
    			//move is invalid and so applied to score
    			worm_owner_list[1].score += invalid_command_score;
    		}
    		
    		//apply shot here
    		if(valid_move1 == 1)
    		{
    			shoot_worm(worm_owner_list[1].next_worm_id,drow,dcol,turn);
    		}
    	}
    }
    
    private void update_graphics()
    {
    	//update worm position and invisibility
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		double alpha = 1;
    		if(worm_list[i].health == 0)
    		{
    			alpha = 0;
    		}
    		
    		worm_mask[i]
					.setX(map_left + worm_list[i].col*mapcell_width)
					.setY(map_top + worm_list[i].row*mapcell_height)
					;
    		
    		worm_sprite[i]
					.setX(map_left + worm_list[i].col*mapcell_width)
					.setY(map_top + worm_list[i].row*mapcell_height)
					.setAlpha(alpha)
					;
    	}
    	
    	//update potion invisibility
    	for(int i = 0;i<num_of_health_packs;i++)
    	{
    		double alpha = 1;
    		if(potion_list[i].strength == 0)
    		{
    			alpha = 0;
    		}
    		
    		potion_mask[i]
    				.setAlpha(alpha);
					;
    		
    		
    		potion_sprite[i]
    				.setAlpha(alpha);
					;
    	}
    	
    	//update map
    	for(int row = 0;row<mapSize;row++)
    	{
    		for(int col = 0;col<mapSize;col++)
    		{
    			String image_file = "";
    			double alpha = 1;
    			if(map[row][col] == 'X')
    			{
    				image_file = "deep_space.png";
    				alpha = 0;
    			}
    			else if(map[row][col] == '.')
    			{
    				image_file = "air.png";
    			}
    			else if(map[row][col] == '#')
    			{
    				image_file = "dirt.png";
    			}
    			
    			map_sprite[row][col]
    					.setImage(image_file)
    					.setAlpha(alpha)
    					;
    		}
    	}
    	
    	//update player score
    	for(int i = 0;i<num_of_players;i++)
    	{
    	    player_score[i].setText( String.valueOf(worm_owner_list[i].score) );
    	}
    	
    	//update worm health bars
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		double alpha = 1;
    		if(worm_list[i].health == 0)
    		{
    			alpha = 0;
    		}
    		
    		for(int f_line = 0;f_line <=1 ;f_line++)
    		{
    			int line_length = mapcell_width;
    			if(f_line == 1)
    			{
    				line_length = worm_list[i].health * mapcell_width / max_health;
    			}
    			
    			health_bars[i][f_line]
    					.setX(map_left + worm_list[i].col*mapcell_width)
    					.setY(map_top + worm_list[i].row*mapcell_height + mapcell_height)
    					.setX2(map_left + worm_list[i].col*mapcell_width + line_length)
    					.setY2(map_top + worm_list[i].row*mapcell_height + mapcell_height)
    					
    					.setAlpha(alpha)
    					;
    		}
    	}
    }
    
    private int dist(int row0,int col0,int row,int col)
    {
    	int dist = (int)Math.sqrt( Math.pow(row0-row, 2) + Math.pow(col0-col, 2));
    	return dist;
    }
    
    private void move_worm(int worm_id,int row,int col)
    {
    	//move worm to target
    	worm_list[worm_id].row = row;
    	worm_list[worm_id].col = col;
    	
    	//apply move score to worms owner
    	worm_owner_list[worm_list[worm_id].owner].score += move_score;
    	
    	//pick up potions on new target
    	for(int i = 0;i<num_of_health_packs;i++)
    	{
    		if(worm_list[worm_id].row == potion_list[i].row && worm_list[worm_id].col == potion_list[i].col)
    		{
    			if(potion_list[i].type.equals("health"))
    			{
    				adjust_worm_health(worm_id,potion_list[i].strength);
    				potion_list[i].strength = 0;
    			}
    		}
    	}
    }

    private void adjust_worm_health(int worm_id,int health_points)
    {
    	worm_list[worm_id].health += health_points;
    	//adjust back to range
    	if(worm_list[worm_id].health < 0)
    	{
    		worm_list[worm_id].health = 0;
    	}
    	if(worm_list[worm_id].health > max_health)
    	{
    		worm_list[worm_id].health = max_health;
    	}
    }
    
    private void dig_worm(int worm_id,int row,int col)
    {
    	//apply move score to worms owner
    	worm_owner_list[worm_list[worm_id].owner].score += dig_score;
    	
    	map[row][col] = '.';
    }
    
    private void shoot_worm(int worm_id,int drow,int dcol,int turn)
    {
    	int target_row = worm_list[worm_id].row + drow;
    	int target_col = worm_list[worm_id].col + dcol;
    	
    	boolean hit = false;
    	while(dist(worm_list[worm_id].row,worm_list[worm_id].col,target_row,target_col) <= worm_list[worm_id].weapon_range && hit == false)
    	{
    		if(0 <= target_row && target_row < mapSize && 0 <= target_col && target_col < mapSize)
    		{
    			//did it hit dirt or space
    			if(map[target_row][target_col] == '#' || map[target_row][target_col] == 'X')
    			{
    				hit = true;
    				
    				//add in graphics for shooting
    				shoot_graphics(worm_id,target_row,target_col,turn);
    				
    				worm_owner_list[worm_list[worm_id].owner].score += missed_attack;
        			
    			}
    			else
    			{
    				//check if it hit any worms
    				for(int i = 0;i<num_of_worms;i++)
    				{
    					if(worm_list[i].row == target_row && worm_list[i].col == target_col && worm_list[i].health > 0)
    					{
    						hit = true;
    						
    						//add in graphics for shooting
    	    				shoot_graphics(worm_id,target_row,target_col,turn);
    	    				
    						adjust_worm_health(i,worm_list[worm_id].weapon_damage*-1);
    						//opponent hit
    						if(worm_list[worm_id].owner != worm_list[i].owner)
    						{
    							worm_owner_list[worm_list[worm_id].owner].score += enemy_worm_hit;
    							if(worm_list[i].health == 0)
    							{
    								worm_owner_list[worm_list[worm_id].owner].score += enemy_worm_kill;
    							}
    						}
    						else //friendly fire
    						{
    							worm_owner_list[worm_list[worm_id].owner].score += own_worm_hit;
    						}
    						
    						i = num_of_worms;
    					}
    				}
    			}
    		}
    		else
    		{
    			//outside of map
    			hit = true;
    			//add in graphics for shooting
				shoot_graphics(worm_id,target_row,target_col,turn);
				
    			worm_owner_list[worm_list[worm_id].owner].score += missed_attack;
    			
    		}
    		
    		target_row += drow;
    		target_col += dcol;
    	}
    	
    	if(hit == false)
    	{
    		worm_owner_list[worm_list[worm_id].owner].score += missed_attack;
    		//add in graphics for shooting
			shoot_graphics(worm_id,target_row,target_col,turn);
			
    	}
    }
    
    private void shoot_graphics(int worm_id,int target_row,int target_col,int turn)
    {
    	int index = (turn%2)*2 + worm_list[worm_id].owner%2;
    	
    	int r = Math.abs(worm_list[worm_id].owner - 1)*250;
    	int g = 0;
    	int b = Math.abs(worm_list[worm_id].owner)*250;
    	int rgb = 65536 * r + 256 * g + b;
    	
    	shooting_lazers[index] = graphicEntityModule.createLine()
    			.setLineWidth(3)
    			.setX(map_left + worm_list[worm_id].col*mapcell_width + mapcell_width/2)
    			.setY(map_top + worm_list[worm_id].row*mapcell_height + mapcell_height/2)
    			.setX2(map_left + target_col*mapcell_width  + mapcell_width/2)
    			.setY2(map_top + target_row*mapcell_height + mapcell_height/2)
    			.setLineColor(rgb)
    			.setZIndex(10)
    			.setAlpha(1);
    }
    
    private void reset_shoot_alpha()
    {
    	for(int i = 0;i<4;i++)
    	{
    		shooting_lazers[i].setAlpha(0);
    	}
    }
    
    private void update_next_worm_id()
    {
    	for(int player_id = 0;player_id < num_of_players;player_id++)
    	{
    		for(int i = 1;i <=num_of_worms;i++)
    		{
    			if(worm_list[ (worm_owner_list[player_id].next_worm_id + i) % num_of_worms].owner == player_id && 
    					worm_list[ (worm_owner_list[player_id].next_worm_id + i) % num_of_worms].health > 0)
    			{
    				worm_owner_list[player_id].next_worm_id = (worm_owner_list[player_id].next_worm_id + i) % num_of_worms;
    				i = num_of_worms+1;
    			}
    		}
    	}
    }
    
    private void check_win_condition()
    {
    	//number of worms left
    	int[] worms_left = new int[num_of_players];
    	for(int i = 0;i<num_of_players;i++)
    	{
    		worms_left[i] = 0;
    	}
    	
    	for(int i = 0;i<num_of_worms;i++)
    	{
    		if(worm_list[i].health > 0)
    		{
    			worms_left[worm_list[i].owner]++;
    		}
    	}
    	
    	boolean endgame = false;
    	for(int i = 0;i<num_of_players;i++)
    	{
    		if(worms_left[i] == 0)
    		{
    			gameManager.getPlayer(i).deactivate();
    			endgame = true;
    		}
    	}
    	
    	if(endgame)
    	{
    		gameManager.endGame();
    	}
    }
}
