package com.codingame.game;

import com.codingame.gameengine.core.AbstractPlayer.TimeoutException;
import com.codingame.gameengine.core.AbstractReferee;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.entities.Sprite;
import com.codingame.gameengine.module.entities.Text;
import com.google.inject.Inject;

public class Referee extends AbstractReferee {

	@Inject private MultiplayerGameManager<Player> manager;
	@Inject private GraphicEntityModule graphics;
	
	private Sprite p1, p2;
	private Text score1, score2;
	
	private Player player1, player2;
	
	@Override
	public void init() {
		int width = graphics.getWorld().getWidth(), height = graphics.getWorld().getHeight();
		
		manager.setTurnMaxTime(50);
		
		player1 = manager.getPlayer(0);
		player2 = manager.getPlayer(1);
		
		graphics.createRectangle()
				.setFillColor(0xffffff)
				.setWidth(width)
				.setHeight(height)
				.setX(0)
				.setY(0);
		
		p1 = graphics.createSprite().setImage("rock.png").setX(width/2-width/15).setY(0);
		p2 = graphics.createSprite().setImage("rock.png").setX(width/2).setY(height).setRotation(Math.PI);
		
		score1 = graphics.createText(player1.getNicknameToken()+": 0");
		score2 = graphics.createText(player2.getNicknameToken()+": 0");
		
		graphics.createSprite()
				.setImage(player1.getAvatarToken())
				.setX(width/2+width/4)
				.setY(0)
				.setBaseHeight(200)
				.setBaseWidth(200);
		graphics.createSprite()
				.setImage(player2.getAvatarToken())
				.setX(width/2+width/4)
				.setY(height-200)
				.setBaseHeight(200)
				.setBaseWidth(200);
		
		score1.setX(10).setY(10).setZIndex(10).setScale(2.5);
		score2.setX(10).setY(80).setZIndex(10).setScale(2.5);
	}

	@Override
	public void gameTurn(int turn) {
		player1.sendInputLine(player2.getPreviousMove() == null ? "None" : player2.getPreviousMove().toString());
		player2.sendInputLine(player1.getPreviousMove() == null ? "None" : player1.getPreviousMove().toString());
		
		Move player1Move = getNextMove(player1), player2Move = getNextMove(player2);
		
		if(player1Move == null || player2Move == null) {
			manager.endGame();
			return;
		}
		
		player1.setPreviousMove(player1Move);
		player2.setPreviousMove(player2Move);
		
		if(player1Move == player2Move) {
			manager.addToGameSummary("Tie!");
		}
		else if(player1Move.getBeats() == player2Move) {
			manager.addToGameSummary(formatWinMessage(player1, player2));
			player1.setScore(player1.getScore()+1);
		}
		else if(player2Move.getBeats() == player1Move) {
			manager.addToGameSummary(formatWinMessage(player2, player1));
			player2.setScore(player2.getScore()+1);
		}
		
		p1.setImage(player1Move+".png");
		p2.setImage(player2Move+".png");
		
		updateScoreCard(player1, score1);
		updateScoreCard(player2, score2);
		
		graphics.commitWorldState(0.0);
	}
	private void updateScoreCard(Player player, Text card) {
		card.setText(player.getNicknameToken()+": "+player.getScore());
	}
	private String formatWinMessage(Player winner, Player loser) {
		return String.format("%s's %s beat %s's %s!", winner.getNicknameToken(), winner.getPreviousMove(), loser.getNicknameToken(), loser.getPreviousMove());
	}
	private Move getNextMove(Player player) {
		String move = null;
		try {
			player.execute();
			move = player.getOutputs().get(0);
			return Move.valueOf(move.toUpperCase());
		}
		catch(TimeoutException e) {
			player.setScore(-1);
			player.deactivate("The program did not output a response in time");
		}
		catch(IllegalArgumentException e) {
			player.setScore(-1);
			player.deactivate("Expected ROCK|PAPER|SCISSORS, found "+move);
		}
		return null;
	}
	@Override
	public void onEnd() {
		updateScoreCard(player1, score1);
		updateScoreCard(player2, score2);
		
		int score1 = player1.getScore();
		int score2 = player2.getScore();
		
		if(score1 == score2) {
			manager.addToGameSummary("Tie!");
		}
		else {
			manager.addToGameSummary((score1 < score2 ? player2 : player1).getNicknameToken()+" wins!");
		}
		graphics.commitWorldState(0.0);
	}
}
