package com.codingame.game;

import java.util.Random;
import java.util.ArrayList;
import java.util.Comparator;

import com.codingame.gameengine.core.GameManager;
import com.codingame.gameengine.core.MultiplayerGameManager;
import com.codingame.gameengine.module.entities.GraphicEntityModule;
import com.codingame.gameengine.module.entities.Rectangle;
import com.codingame.gameengine.module.entities.Sprite;
import com.codingame.gameengine.module.entities.SpriteAnimation;
import com.codingame.gameengine.module.entities.Text;
import com.codingame.gameengine.module.entities.TextBasedEntity;
import com.codingame.gameengine.module.entities.Group;
import com.codingame.gameengine.module.entities.Curve;
import com.google.inject.Inject;

class View {
    @Inject private MultiplayerGameManager<com.codingame.game.Player> gameManager;
    @Inject private GraphicEntityModule graphicEntityModule;
    @Inject PantsModule pantsModule;

    /*
     * Frame timings, for a base frame length of 2s:
     *   - first half: stone throw
     *   - second half: troll move
     * The troll message is anchored around the troll move.
     */
    private final double STONE_THROW_START = 0.0;
    private final double STONE_THROW_PEAK = 0.25;
    private final double STONE_THROW_END = 0.5;
    private final double TROLL_MOVE_START = 0.5;
    private final double TROLL_MOVE_END = 1.0;
    private final double TROLL_MESSAGE_START = 0.5;
    private final double TROLL_MESSAGE_END = 1.0;

    /*
     * Castle destruction and endgame message pertain to an endgame
     * frame only.
     */
    private final double AVATAR_ANIMATION_START = 0.5;
    private final double CASTLE_DESTRUCTION_START = 0.0;
    private final double CASTLE_DESTRUCTION_END = 0.5;

    class Player {
        Model.Player model;

        int colorToken;
        String nicknameToken;
        String avatarToken;

        double frameRot;

        Group avatar;
        Text stoneCounter;
        Text message;
        Sprite castle;
        Text stone;
        Text stoneReminder;

        void init(com.codingame.game.Player p) {
            model = p.model;
            colorToken = p.getColorToken();
            nicknameToken = p.getNicknameToken();
            avatarToken = p.getAvatarToken();

            boolean p0 = model.index == 0;
            int x = p0 ? 280 : 1920 - 280;
            int y = 220;

            Sprite frame = graphicEntityModule.createSprite()
                .setImage("frame.png")
                .setAnchor(0.5)
                .setRotation(frameRot)
                .setZIndex(22)
                .setTint(colorToken);

            Sprite frameBg = graphicEntityModule.createSprite()
                .setImage("frame_bg.png")
                .setAnchor(0.5)
                .setRotation(frameRot)
                .setZIndex(20);

            Sprite avatarSprite = graphicEntityModule.createSprite()
                .setZIndex(21)
                .setImage(avatarToken)
                .setAnchor(0.5)
                .setBaseHeight(116)
                .setBaseWidth(116);

            avatar = graphicEntityModule
                .createGroup(frame, frameBg, avatarSprite)
                .setX(x).setY(y);

            Text text = graphicEntityModule.createText(nicknameToken)
                    .setX(x)
                    .setY(y + 120)
                    .setZIndex(20)
                    .setFontSize(40)
                    .setFillColor(0x7f3f00)
                    .setAnchor(0.5);

            stoneCounter = graphicEntityModule.createText()
                .setX(x)
                .setY(y+200)
                .setZIndex(20)
                .setFontSize(40)
                .setFillColor(0x7f3f00)
                .setAnchor(0.5);
            updateStoneCounter();

            message = graphicEntityModule.createText()
                .setX(p0 ? 15 : 1920-15)
                .setY(680)
                .setZIndex(1)
                .setFontSize(40)
                .setFillColor(0xffbf7f)
                .setAnchorX(p0 ? 0 : 1)
                .setAnchorY(1);

            castle = graphicEntityModule.createSprite()
                .setImage("castle.png")
                .setTint(colorToken)
                .setX(p0 ? 160 : 1920-160)
                .setY(p0 ? 890 : 880)
                .setZIndex(1)
                .setAnchorX(0.5)
                .setAnchorY(1)
                .setScaleX(p0 ? 1 : -1);

            stone = graphicEntityModule.createText()
                .setZIndex(3)
                .setFontSize(150)
                .setFillColor(0x12322a)
                .setAnchor(0.5)
                .setAlpha(0);

            stoneReminder = graphicEntityModule.createText()
                .setX(p0 ? x + 100 : x - 100)
                .setY(y)
                .setZIndex(20)
                .setFontSize(80)
                .setFontFamily("monospace")
                .setStrokeColor(0xff0080)
                .setFillColor(0xff0080)
                .setAnchorX(p0 ? 0 : 1)
                .setAnchorY(0.5);
            pantsModule.displayOnToggleState(stoneReminder, "debug", true);
        }

        void startTurn() {
            graphicEntityModule.commitEntityState(0.0, stoneReminder);
        }

        void victory() {
            gameManager.addToGameSummary(GameManager.formatSuccessMessage(nicknameToken + " wins."));
            View.this.endgameFrame();
            markWinner();
        }

        void defeat() {
            gameManager.addToGameSummary(GameManager.formatErrorMessage(trollRace.starter + " destroys " + nicknameToken + "."));
            destroyCastle();
        }

        // ========== Player/avatar markings

        void markTimeout() {
            animateLoss(avatar.getX(), avatar.getY(), 100, "SLOW\nPOKE");
        }

        void markIllegal() {
            animateLoss(avatar.getX(), avatar.getY(), 100, "STUPID");
        }

        void markCheat() {
            animateLoss(avatar.getX(), avatar.getY(), 100, "CHEATER");
        }

        void markWinner() {
            graphicEntityModule.commitEntityState(AVATAR_ANIMATION_START, avatar);
            avatar.setScaleX(1.5, Curve.EASE_OUT);
            avatar.setScaleY(1.5, Curve.EASE_OUT);
            avatar.setRotation((random.nextDouble() - 0.5) * Math.PI / 18,
                               Curve.ELASTIC);
        }

        void markLoser() {
            graphicEntityModule.commitEntityState(AVATAR_ANIMATION_START, avatar);
            int dir = random.nextInt(2) == 1 ? 1 : -1;
            avatar.setRotation(dir * 170 * Math.PI / 180, Curve.ELASTIC);
        }

        // ==========Player/stones

        void throwStones(int stones) {
            gameManager.addToGameSummary(String.format("%s throws %d stone%s at " + trollRace.nonStarter(), nicknameToken, stones, stones == 1 ? "" : "s"));
        }

        void threwMoreStonesThanHad() {
            gameManager.addToGameSummary(GameManager.formatErrorMessage(nicknameToken + " tried to throw more stones than they had.  I'll let it slide for this time.  (But not let them throw that much!)"));
        }

        void failedToThrowStonesAndShouldHave() {
            gameManager.addToGameSummary(GameManager.formatErrorMessage(nicknameToken + " tried not throwing any stones.  Fixing that for them because I'm in a good mood today."));
        }

        void updateStoneCounter() {
            int stones = model.getStones();
            if (stones <= 0) {
                stoneCounter.setText("Out of stones!");
                stoneCounter.setFillColor(0xff7777);
            }
            else if (stones == 1) {
                stoneCounter.setText("1 stone");
                stoneCounter.setFillColor(0xffbb77);
            }
            else {
                stoneCounter.setText(stones + " stones");
            }
            graphicEntityModule.commitEntityState(STONE_THROW_PEAK, stoneCounter);
        }

        void animateStones(int stones) {
            String stonesString = Integer.valueOf(stones).toString();
            stone.setX(castle.getX());
            stone.setY(castle.getY() - 100);
            stone.setText(stonesString);
            stone.setAlpha(1);
            graphicEntityModule.commitEntityState(STONE_THROW_START, stone);
    
            int peakX = (castle.getX() + troll.getX()) / 2;
            int peakY = 540;
            stone.setX(peakX);
            stone.setY(peakY, Curve.EASE_OUT);
            graphicEntityModule.commitEntityState(STONE_THROW_PEAK,
                                                  stone,
                                                  stoneCounter);
    
            stone.setX(troll.getX());
            stone.setY(troll.getY() - 50, Curve.EASE_IN);
            stone.setAlpha(0, Curve.EASE_IN);
            graphicEntityModule.commitEntityState(STONE_THROW_END, stone);

            stoneReminder.setText(stonesString);
            graphicEntityModule.commitEntityState(0.0, stoneReminder);
        }

        // ========== Player/castle

        void displayMessage(String msg) {
            message.setText(msg);
            graphicEntityModule.commitEntityState(0.0, message);
        }

        void destroyCastle() {
            graphicEntityModule.commitEntityState(CASTLE_DESTRUCTION_START, castle);
            castle.setX(castle.getX(), Curve.ELASTIC);
            castle.setScaleY(-0.2, Curve.EASE_IN);
            graphicEntityModule.commitEntityState(CASTLE_DESTRUCTION_END, castle);
        }
    } // class Player

    Model model;
    Random random = new Random();
    Text trollMessage;
    Group troll;
    Text trollPositionGauge;
    Player p0 = new Player(), p1 = new Player();
    Text turnCounter; int _turns = 0;

    // ==================== Referee interface

    void init(Model m) {
        model = m;
        drawBackground();

        /*
         * Random π/2-grained rotation of the avatar frames.  Avoid
         * having them π/2 apart, though, as one of them is likely
         * going to end upside-down and the trick would be revealed.
         * And I'd have to "draw" a new frame. Ewww.
         */
        p0.frameRot = random.nextInt(4) * Math.PI / 2;
        p0.init(gameManager.getPlayer(0));
        p1.frameRot = p1.frameRot +
            (random.nextInt(2) == 1 ? 1 : -1) * Math.PI / 2;
        p1.init(gameManager.getPlayer(1));

        drawTroll();

        drawDebug();
    }

    void startTurn() {
        p0.startTurn();
        p1.startTurn();

        trollMessage.setX(troll.getX());

        animateTurnCounter();
    }

    void endgameFrame() {
        gameManager.setFrameDuration(2000);
    }

    void doubleDefeat() {
        gameManager.addToGameSummary(GameManager.formatErrorMessage("Everybody loses!"));
        endgameFrame();
        animateLoss(1920/2, 680, 150, "L0SERZ!");
    }

    void draw() {
        gameManager.addToGameSummary("Draw.");
        endgameFrame();
        animateLoss(1920/2, 680, 200, "DRAW");
    }

    // drawBackground() helper class
    // @Java nerds: is this avoidable?
    private class Pos {
        int x, y;
        Pos(int _x, int _y) { x = _x; y = _y; }
    }

    private void drawBackground() {
        graphicEntityModule.createSprite()
                .setImage("background.png")
                .setAnchor(0);

        int numMountains = random.nextInt(5);
        while (numMountains --> 0) {
            final int pngWidth = 366;
            double scale = 0.5 * (1 + random.nextDouble());
            int x = random.nextInt(1920 + (int) (scale*pngWidth))
                - (int) (scale*pngWidth/2);
            int baseTint = 64 + random.nextInt(128);
            Sprite mountain = graphicEntityModule.createSprite()
                .setImage("mountain.png")
                .setX(x)
                .setY(680)
                .setAnchorX(0.5)
                .setAnchorY(283.0 / 321.0)
                .setRotation((random.nextDouble() - 0.5) * Math.PI / 1800)
                .setScaleX(random.nextInt(2) == 0 ? scale : -scale)
                .setScaleY(scale * (1 + (random.nextDouble() - 0.5) / 2))
                .setSkewX((random.nextDouble() - 0.5) / 4)
                .setSkewY((random.nextDouble() - 0.5) / 8)
                .setTint((baseTint + random.nextInt(16) - 8) * 0x010000
                         + (baseTint + random.nextInt(16) - 8) * 0x0100
                         + (baseTint + random.nextInt(16) - 8) * 0x01);
            graphicEntityModule.createSprite().setImage("mountaintop.png")
                .setX(mountain.getX())
                .setY(mountain.getY())
                .setAnchorX(mountain.getAnchorX())
                .setAnchorY(mountain.getAnchorY())
                .setRotation(mountain.getRotation())
                .setScaleX(mountain.getScaleX())
                .setScaleY(mountain.getScaleY())
                .setSkewX(mountain.getSkewX())
                .setSkewY(mountain.getSkewY());
        }

        int numTrees = random.nextInt(21);
        ArrayList<Pos> poss = new ArrayList<Pos>(numTrees);
        while (numTrees --> 0) {
            int x, y;
            do {
                x = random.nextInt(1920);
                // yes, this biases randomness wrt perspective! :-(
                y = 700 + random.nextInt(175);
            } while (y > 880 && (x < 200 || x > 1720));
            poss.add(new Pos(x, y));
        }
        poss.sort(new Comparator<Pos>() {
                public int compare(Pos a, Pos b) { return a.y < b.y ? -1 : 1; }
            });

        for (Pos p : poss) {
            double scale = ( 90.0 / 433.0           // base height from PNG
                             * (p.y - 680) / (875 - 680) ); // perspective
            graphicEntityModule.createSprite()
                .setImage(random.nextInt(2) == 0 ? "Alshockv1.png"
                                                 : "Alshockv2.png")
                .setAnchorX(0.5)
                .setAnchorY(1)
                .setX(p.x)
                .setY(p.y)
                .setScaleX(scale * (random.nextInt(2) == 0 ? -1 : 1)
                           * (1 + (random.nextDouble() - 0.5) / 6))
                .setScaleY(scale * (1 + (random.nextDouble() -0.5) / 6))
                .setRotation((random.nextDouble() - 0.5) * Math.PI / 1800)
                .setSkewX((random.nextDouble() - 0.5) /4)
                .setSkewY((random.nextDouble() - 0.5) /8);
        }

        // base png: 514×387
        Sprite f7u12 = graphicEntityModule.createSprite()
            .setImage("f7u12.png")
            .setX(1920 / 2)
            .setY(1080 / 2)
            .setAnchorX(0.5)
            .setAnchorY(0.5)
            .setBaseWidth(514*1080/387)
            .setBaseHeight(1080)
            .setZIndex(200);
        pantsModule.displayOnToggleState(f7u12, "troll", true);
    }

    // ==================== Troll

    enum TrollRace {
        Troll("The troll", 0xfac200, "bland"),
        IceTroll("The ice troll", 0x59a2a2, "ice"),
        RockTroll("The rock troll", 0x78877f, "rock"),
        WaterTroll("The water troll", 0x2b2fc6, "water"),
        OlogHai("The Olog-Hai", 0x5b2e7d, "ologhai");
        String starter, parser; int tint;
        TrollRace(String s, int t, String p) {
            starter = s;
            tint = t;
            parser = p;
        }
        String nonStarter() {
            return Character.toLowerCase(starter.charAt(0))
                + starter.substring(1);
        }
    }
    TrollRace trollRace;

    private void drawTroll() {
        int r, league = gameManager.getLeagueLevel();
        if (league <= 1) r = 4;
        else if (league <= 2) r = 8;
        else r = 10;

        r = random.nextInt(r);
        if (r < 4) trollRace = TrollRace.Troll;
        else if (r < 6) trollRace = TrollRace.IceTroll;
        else if (r < 8) trollRace = TrollRace.RockTroll;
        else if (r < 9) trollRace = TrollRace.WaterTroll;
        else if (r < 10) trollRace = TrollRace.OlogHai;
        else throw new RuntimeException("Internal error: unknown troll race " + r);

        // We read it for debugging purposes, but don't echo it back
        // to the IDE.  It is, after all, *not* a map parameter!
        String buf = gameManager.getGameParameters().getProperty("ehtnicity");
        if (buf != null) {
            String key = "";
            for (char c : buf.toCharArray())
                if (Character.isLetter(c))
                    key += Character.toLowerCase(c);
            iHateJava: do {
                for (TrollRace race : TrollRace.values()) {
                    if (key.equals(race.parser)) {
                        trollRace = race;
                        break/*ing news: */ iHateJava;
                    }
                }
                gameManager.addToGameSummary("Ignoring unknown troll race: " + buf);
            } while (false);
        }
        photoFinish: ; // The race is through, but Java has no goto :-(

        Sprite trollBody = graphicEntityModule.createSprite()
            .setImage("troll_body.png")
            .setAnchorX(0.5)
            .setAnchorY(1)
            .setTint(trollRace.tint);
        Sprite trollPantsRed = graphicEntityModule.createSprite()
            .setImage("pants_red.png")
            .setAnchorX(0.5)
            .setAnchorY(1);
        pantsModule.displayOnPantsState(trollPantsRed, 1);
        Sprite trollPantsGreen = graphicEntityModule.createSprite()
            .setImage("pants_green.png")
            .setAnchorX(0.5)
            .setAnchorY(1);
        pantsModule.displayOnPantsState(trollPantsGreen, 2);
        Sprite trollPantsBlue = graphicEntityModule.createSprite()
            .setImage("pants_blue.png")
            .setAnchorX(0.5)
            .setAnchorY(1);
        pantsModule.displayOnPantsState(trollPantsBlue, 3);
        Sprite trollPantsPerv = graphicEntityModule.createSprite()
            .setImage("pants_perv.png")
            .setAnchorX(0.5)
            .setAnchorY(1);
        pantsModule.displayOnPantsState(trollPantsPerv, 4);
        troll = graphicEntityModule
            .createGroup(trollBody, trollPantsRed,
                         trollPantsGreen, trollPantsBlue, trollPantsPerv)
            .setX(1920/2)
            .setY(880)
            .setScaleX(random.nextInt(2) == 0 ? 1 : -1)
            .setZIndex(2);
        trollPositionGauge = graphicEntityModule.createText()
            .setZIndex(2)
            .setAnchor(0.5)
            .setFontSize(40)
            .setX(1980/2)
            .setY(980)
            .setFillColor(0xffffff);
        moveTroll();

        trollMessage = graphicEntityModule.createText()
            .setZIndex(1)
            .setX(1902/2)
            .setY(680)
            .setAnchorX(0.5)
            .setAnchorY(0)
            .setTextAlign(TextBasedEntity.TextAlign.CENTER)
            .setStrokeColor(0xFFFF00)
            .setFillColor(0xFFFF00)
            .setFontSize(40);
        pantsModule.displayOnToggleState(trollMessage, "verboseTrolling", true);
    }

    void moveTroll() {
        graphicEntityModule.commitEntityState(TROLL_MOVE_START, troll, trollPositionGauge);
        int x0 = p0.castle.getX(), x1 = p1.castle.getX();
        int y0 = p0.castle.getY(), y1 = p1.castle.getY();
        troll.setX(x0 + model.trollPosition * (x1-x0) / model.roadLength,
                   Curve.ELASTIC);
        troll.setY(y0 + model.trollPosition * (y1-y0) / model.roadLength,
                   Curve.ELASTIC);

        trollPositionGauge.setX((trollPositionGauge.getX() + troll.getX()) / 2);
        int distLeft = model.trollPosition;
        int distRight = model.roadLength - model.trollPosition;
        if (distLeft <= 0) {
            trollPositionGauge.setText("← " + distRight);
        }
        else if (distRight <= 0) {
            trollPositionGauge.setText(distLeft + " →");
        }
        else {
            trollPositionGauge.setText(distLeft + " ↔ " + distRight);
        }
        final double moveMid = (TROLL_MOVE_START + TROLL_MOVE_END) / 2;
        graphicEntityModule.commitEntityState(moveMid, trollPositionGauge);
        trollPositionGauge.setX(troll.getX());
    }

    enum Dir {
        LEFT("walks left.", 0),
        STILL("stands still.", 1),
        RIGHT("walks right.", 2);

        String movement; int index;
        Dir(String mvt, int i) { movement = mvt; index = i; }
    }

    void moveTroll(Dir d) {
        moveTroll();
        gameManager.addToGameSummary(trollRace.starter + " " + d.movement);

        trollMessage.setText(selectTrollMessage(d)).setAlpha(1, Curve.NONE);
        graphicEntityModule.commitEntityState(TROLL_MESSAGE_START, trollMessage);
        trollMessage.setAlpha(0, Curve.EASE_IN);
        graphicEntityModule.commitEntityState(TROLL_MESSAGE_END, trollMessage);
    }

    String selectTrollMessage(Dir d) {
        if (random.nextInt(10000) == 0) {
            return TrollText.specials[random.nextInt(TrollText.specials.length)];
        }

        // yup, still biased
        int i = random.nextInt(TrollText.directed.length + TrollText.isotropic.length / 3);
        if (i < TrollText.directed.length) {
            return TrollText.directed[i][d.index];
        }
        else {
            return TrollText.isotropic[random.nextInt(TrollText.isotropic.length)];
        }
    }

    // ==================== Debug information

    void animateTurnCounter() {
        for (int i = 0; i < 10; i++) {
            turnCounter.setText("T" + _turns + "." + i);
            // The following line is likely not a bug.
            graphicEntityModule.commitEntityState((double) i/9, turnCounter);
        }
        _turns++;
    }

    void drawDebug() {
        String[] debugModePngs = graphicEntityModule.createSpriteSheetSplitter()
            .setSourceImage("debug.png")
            .setImageCount(2)
            .setWidth(900)
            .setHeight(150)
            .setOrigRow(0)
            .setOrigCol(0)
            .setImagesPerRow(1)
            .setName("debug")
            .split();
        SpriteAnimation debugMode = graphicEntityModule.createSpriteAnimation()
            .setImages(debugModePngs)
            .setX(1920 / 2)
            .setY(60)
            .setAnchorX(0.5)
            .setLoop(true);
        pantsModule.displayOnToggleState(debugMode, "debug", true);

        turnCounter = graphicEntityModule.createText()
            .setAnchorX(0.5)
            .setAnchorY(0)
            .setX(1920 / 2)
            .setY(260)
            .setStrokeColor(0xff0080)
            .setFillColor(0xff0080)
            .setFontFamily("monospace")
            .setFontWeight(Text.FontWeight.BOLD)
            .setFontSize(100);
        pantsModule.displayOnToggleState(turnCounter, "debug", true);
        animateTurnCounter();
    }

    // ==================== Endgame status

    void animateLoss(int x, int y, int size, String message) {
        int startX;
        if (x < 1920/2) { startX = 1920; }
        else if (x > 1920/2) { startX = 0; }
        else { startX = 1920 * random.nextInt(2); }

        Text msg = graphicEntityModule.createText(message)
            .setX(startX)
            .setY(1080)
            .setAnchorX(0.5)
            .setAnchorY(0.5)
            .setScaleX(3*random.nextDouble() - 1)
            .setScaleY(3*random.nextDouble() - 1)
            .setSkewX(2*random.nextDouble() - 1)
            .setSkewY(2*random.nextDouble() - 1)
            .setRotation(4*Math.PI * (1 + random.nextDouble())
                         * (random.nextInt(2) == 0 ? 1 : -1))
            .setFontSize(0)
            .setStrokeColor(0xff7f7f)
            .setFillColor(0xff7f7f)
            .setFontWeight(Text.FontWeight.BOLD)
            .setTextAlign(TextBasedEntity.TextAlign.CENTER);
        graphicEntityModule.commitEntityState(0.0, msg);
        Curve curve = Curve.ELASTIC;
        msg.setX(x, Curve.EASE_OUT)
            .setY(y, Curve.ELASTIC)
            .setScaleX(1, curve)
            .setScaleY(1, curve)
            .setSkewX(0, curve)
            .setSkewY(0, curve)
            .setRotation(2*Math.PI * (random.nextDouble() - 0.5), Curve.LINEAR)
            .setFontSize(size, curve);
    }
}
